/*  gen_corpora.c - write out initial fuzz data for zint, allowing for how `set_symbol()` in "fuzz.h" uses first X
    bytes to set various `zint_symbol` members (namely `input_mode`, `option_1`, `option_2`, `eci`, `option_3`) */
/*
    From "backend/tests/fuzz" directory:

	rm -rf fuzz_data_corpus fuzz_gs1_corpus
	rm -f fuzz_data_seed_corpus.zip fuzz_gs1_seed_corpus.zip
	gcc gen_corpora.c -o gen_corpora -lzint -fsanitize=address
	./gen_corpora
	zip -j fuzz_data_seed_corpus.zip fuzz_data_corpus/*
	zip -j fuzz_gs1_seed_corpus.zip fuzz_gs1_corpus/*
*/
/*
    libzint - the open source barcode library
    Copyright (C) 2024 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <errno.h>
#include <unistd.h>
#include <sys/stat.h> /* mkdir(2) */

#include "fuzz.h"

/* Data item */
struct item {
    int test_fuzz_idx;
    int symbology;
    int input_mode;
    int eci;
    int option_1;
    int option_2;
    int option_3;
    char *data;
    int length;
};

/* DATA_MODE data */
static const struct item data_data[] = {
    /*  0*/ { 0, BARCODE_CODEONE, -1, -1, -1, -1, -1, "3333P33B\035333V3333333333333\0363", -1 }, /* #181 Nico Gunkel, OSS-Fuzz */
    /*  1*/ { 1, BARCODE_CODEONE, -1, -1, -1, -1, -1, "{{-06\024755712162106130000000829203983\377", -1 }, /* #232 Jan Schrewe, CI-Fuzz, out-of-bounds in is_last_single_ascii() sp + 1 */
    /*  2*/ { 2, BARCODE_CODEONE, -1, -1, -1, -1, -1, "\000\000\000\367\000\000\000\000\000\103\040\000\000\244\137\140\140\000\000\000\000\000\000\000\000\000\005\000\000\000\000\000\165\060\060\060\060\061\060\060\114\114\060\010\102\102\102\102\102\102\102\102\057\102\100\102\057\233\100\102", 60 }, /* #300 (#4) Andre Maute (`c1_c40text_cnt()` not accounting for extended ASCII shifts) */
    /*  3*/ { 3, BARCODE_CODEONE, -1, -1, -1, 10, -1, "\153\153\153\060\001\000\134\153\153\015\015\353\362\015\015\015\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\015\015\015\015\015\015\015\015\015\015\015\015\015\015\015\362\362\000", 65 }, /* #300 (#8) Andre Maute (`c1_encode()` looping on latch) */
    /*  4*/ { 4, BARCODE_CODEONE, -1, -1, -1, 10, -1, "\015\015\353\362\015\015\015\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\110\015\015\015\015\015\015\015\015\015\015\015\015\015\015\015\362\362\000", 39 }, /* #300 (#8 shortened) Andre Maute */
    /*  5*/ { 5, BARCODE_CODEONE, -1, -1, -1, 10, -1, "\153\153\153\153\153\060\001\000\000\134\153\153\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\153\153\153\153\153\153\043\000\000\307\000\147\000\000\000\043\113\153\162\162\215\220", 90 }, /* #300 (#12) Andre Maute (too small buffer for Version T) */
    /*  6*/ { 0, BARCODE_TELEPEN, -1, -1, -1, -1, -1, "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000", 69 }, /* #181 Nico Gunkel OSS-Fuzz */
    /*  7*/ { 2, BARCODE_TELEPEN_NUM, -1, -1, -1, -1, -1, "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000", 70 },
    /*  8*/ { 3, BARCODE_TELEPEN_NUM, -1, -1, -1, -1, -1, "0404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404040404", 136 },
    /*  9*/ { 5, BARCODE_TELEPEN_NUM, -1, -1, -1, -1, -1, "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000X", 136 },
    /* 10*/ { 6, BARCODE_TELEPEN_NUM, -1, -1, -1, -1, -1, "9999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999", 136 },
    /* 11*/ { 0, BARCODE_CODABLOCKF, -1, -1, -1, -1, -1, "\034\034I", 3 }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 12*/ { 0, BARCODE_AZTEC, -1, -1, -1, -1, -1,
                "\133\060\060\060\135\060\125\125\125\125\140\060\125\125\125\125\060\060\060\271\060\060\125\103\164\125\125\125\377\377\125\125"
                "\125\125\125\125\125\133\060\076\060\135\261\177\261\261\261\236\261\261\261\040\261\261\261\261\261\261\261\020\261\261\261\261"
                "\261\261\265\261\261\261\261\261\261\261\261\261\261\261\261\040\224\261\261\261\261\261\000\000\004\000\031\060\031\031\031\031"
                "\031\031\031\031\261\261\040\261\261\261\261\261\261\261\020\261\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\376\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\261\261\261\261\261\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\135\135\135\135\135\135"
                "\135\335\135\060\060\010\010\010\010\010\060", 2251
            }, /* Original OSS-Fuzz triggering data for malloc leak */
    /* 13*/ { 1, BARCODE_AZTEC, -1, -1, -1, -1, -1,
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\000\060\060\060\060\000\060\060\000\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                "\060\005\060\060\060\060\060\060\060\005\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\000\000\060\350\060\060\060\060\060\350\060\345\060\343\060\060\060\005\060\060\060"
                "\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060"
                "\060\060\060\060\060\175\175\175\175\060\175\175\175\175\175\060\060\060\060\175\060\175\175\175\175\060\060\060\060\005\060\060"
                "\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\000\060\060\060\060\060\005\060\060\060\060\060\005\005\060\005\060\060\060\060\060\060\005"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\353\060\060\353"
                "\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060"
                "\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060"
                "\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353"
                "\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                "\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                "\060\060\353\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                "\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060"
                "\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060"
                "\353\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353"
                "\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\264\060"
                "\060\060\060\060\264\060\060\060\060\060\264\060\060\060\060\264\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                "\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                "\353\060\060\060\060\060\353\060\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353"
                "\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                "\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060"
                "\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                "\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060"
                "\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060"
                "\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                "\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353"
                "\060\060\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060"
                "\060\060\060\060\353\060\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060"
                "\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060"
                "\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353"
                "\060\060\060\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\343\060\060\060\060\060\060\060\060\060\005\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\377\060\060\060\060\060\060\060"
                "\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\005\060\060\060\060"
                "\060\060\060\060\060\060\060\060\361\060\060\060\060\060\361\060\060\060\060\361\060\060\060\060\060\361\060\060\060\060\060\361"
                "\060\060\060\361\060\060\060\005\060\060\060\060\060\060\060\005\060\060\060\060\060\005\363\060\362\060\060\060\060\377\060\060"
                "\060\177\060\060\060\060\060\263\060\060\060\060\060\060\060\060\060\000\060\060\060\060\377\177\060\060\060\362\060\362\060\363"
                "\060\363\060\177\060\377\060\060\060\060\060\060\060\377\177\060\355\060\057\060\060\060\377\060\377\060\060\060\177\000\060\000"
                "\060\377\060\060\060\060\060\060\005\060\060\000\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\005\060\060"
                "\060\060\060\060\060\371\060\060\060\363\060\060\060\060\060\060\362\060\060\060\060\060\362\060\363\060\362\060\060\060\060\377"
                "\060\177\060\060\060\060\060\263\060\060\060\060\060\060\060\060\060\060\000\060\060\060\377\177\060\060\060\362\060\362\060\363"
                "\060\363\060\060\377\057\060\060\060\060\060\060\377\177\060\355\060\060\060\060\060\377\060\377\060\060\060\177\000\060\000\060"
                "\377\060\353\060\060\000\060\060\060\060\060\060\060\060\060\256\060\060\060\362\060\060\060\060\060\060\060\060\005\362\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\057\060\060\060\060\060\060\060\000\060\000\060\060\060\000\060\060"
                "\057\060\060\060\060\060\377\060\377\060\060\060\060\060\060\060\060\060\060\060\060\000\060\060\347\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\000\060\377\060\060\060\377\060\363\060\363\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                "\362\060\060\060\060\060\060\362\347\060\354\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\005\060\060\060\000\060\000\060\060\060\060"
                "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\344\005\060\060\005\060\342\060"
                "\060\060\060\060\060\060\060\060\060\060\060\005\060\000\060\060\060\060\060\060\060\060\060\060\377\060\060\060\060\005\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\005\060\060\060\060"
                "\060\060\060\060\060\060\060\005\060\060\060\005\060\060\060\060\060\060\060\060\060\060\005\060\351\060\060\060\060\060\060\005"
                "\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005"
                "\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\005\060\060\060\060\005"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\175\175\060\175\060\060\060"
                "\060\175\060\060\060\060\175\060\060\060\060\175\060\175\175\175\005\060\060\060\060\060\060\060\060\377\060\060\060\060\060\060"
                "\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\342\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\005\060\060\060\345\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\005\060\371\060\060\060\060\060\060\060\060\060\060\060\361\060\060\060\060\060\005\361\060\060\060\060"
                "\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\005\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\351\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\005\363\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060\005\060\000\060"
                "\060\060\060\060\060\060\362\060\060\060\060\060\060\060\060\060\060\060\005\060\005\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\000\060\060\060\060\060\060\060\060\060\005\060\377\060\005"
                "\060\371\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\005\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060"
                "\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\175\175\175\175\060\175\175\060\060\060\060\060\175\060\175\175"
                "\175\005\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\005\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\060\060\060\060\060"
                "\060\060\005\060\060\060\060\060\005\060\000\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060"
                "\060\060\005\060\060\060\060\060\005\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                "\060\060\005\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\005\060\351\060\060\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060"
                "\060\005\060\060\060\005\060\362\060\060\060\060\060\060\060\060\060\060\060\060\377\060\060\000\177\060\060\000\060\377\060\353"
                "\060\060\060\363\060\060\060\060\060\060\060\060\060\060\060\060\362\060\060\060\060\060\000\060\060\377\060\060\060\175\175\175"
                "\175\060\060\060\175\175\175\175\060\060\005\060\005\060\005\060\060\060\060\000\000\060\060\060\060\060\060\377\060\060\060\060"
                "\377\060\377\377\060\060\057\060\060\057\060\060\060\000\000\060\060", 2801
            }, /* Original OSS-Fuzz triggering data for binary_string buffer overrun */
    /* 14*/ { 2, BARCODE_AZTEC, -1, -1, 1, -1, -1,
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123",
                4483
            }, /* 4483 = (1664 (Max codewords) - 169 (ECC codewords) - 5/12 (D/L) - 3/12 (padding)) * 3 (3 4-bit digits per 12-bit wordcode) = 4483 */
    /* 15*/ { 8, BARCODE_AZTEC, -1, -1, -1, 1, -1,
                "\105\105\000\000\000\000\000\000\000\000\077\012\377\377\377\072\376\376\350\350\350\350\350\250\350\350\350\350\354\350\350\350\350\350\001\000\000\000\000\000"
                "\000\036\103\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\002\222\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\012"
                "\377\377\377\072\376\376\350\350\350\350\350\250\350\350\350\350\354\350\350\350\000\000\000\000\000\000\000\033\000\036\103\012\072\103\103\000\100\116\000\000"
                "\000\000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\012\000\000\000\000\072\103\103\103\103\012\012\000\365\365\365\365\365\365\374\365\365\365\365\365\000\000\001\000\000\000\000\000\100\377\337\377"
                "\377\377\377\377\000\000\000\000\372\377\000\100\377\377\350\350\000\000\350\350\350\350\350\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\365\000"
                "\000\000\000\000\000\000\000\000\377\377\377\350\350\350\350\350\350\350\350\350\350\350\350\350\350\061\350\350\350\350\354\350\350\350\350\350\001\000\000\000"
                "\000\000\000\036\103\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\216\000\000\000\000\000\000\000\377\377\377\377\377\377\377\000\000"
                "\377\365\374\365\365\365\365\001\236\365\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\064\064\064\064\064\064\064\064\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153"
                "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\064\064\064\064\064\064\064\064\064\064\044\064\064\064\064\064\064\064\064\064"
                "\064\064\064\064\064\064\064\064\064\064\064\064\064\064\064\064\064\264\264\362\362\362\362\242\242\242\242\242\242\242\242\242\242\242\242\242\242\242\242\242"
                "\242\242\242\242\242\242\242\242\242\242\242\242\242\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\012\000\000\000\000\072\103\103\103\103\012\012\000\365\365\365\365\365"
                "\365\374\365\365\365\365\365\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\100\377\377\377\377\377\377\377\000\000\000\000"
                "\000\000\000\100\377\377\350\350\000\000\350\350\350\350\350\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\365\000\000\000\000\000\000\000\266\266"
                "\266\266\112\000\000\000\266\266\266\266\266\266\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\022\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\012\377"
                "\377\377\072\376\376\350\350\350\350\350\000\000\000\000\001\000\000\000\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\100\116\000\000\000"
                "\000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\000\000\000\025\000\000\000\000\001\000\000\000\000\000\000\003\000\000\000\000\000\000\000\000"
                "\000\000\012\000\000\000\000\072\103\103\103\103\012\012\000\365\365\365\365\365\365\374\365\365\365\365\365\000\000\000\311\000\000\000\000\100\377\337\377\377"
                "\377\377\377\000\000\000\000\000\000\000\100\377\377\350\353\000\000\350\150\350\350\350\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\365\000\000"
                "\000\000\000\000\000\047\000\377\377\377\350\350\350\350\350\350\350\350\350\350\350\350\350\350\254\350\350\350\350\354\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\350\350\350\350\350\001\000\000\127\000\000\000\036\103"
                "\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\220\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\050\050\050\000\000\000"
                "\000\000\000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\000\000\000\000\000\000\000\000\000\000\000\000\266"
                "\266\266\266\266\266\377\377\377\377\377\013\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\005\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\266\266\266\266\266\266\266\266\266\266\377\377\377\377\377\377\377\377\377\377\377\377\044"
                "\377\377\377\377\377\377\050\064\064\064\000\000\000\000\072\376\376\350\350\350\350\350\350\377\377\377\377\377\377\377\377\377\377\377\377\377\005\377\377\377"
                "\377\350\350\350\350\350\350\350\310\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\000\000\000\000\000\000\000\000\000\000\000\000\000\266\266"
                "\266\266\266\266\377\377\377\377\377\013\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\005\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\266\266\266\266\266\266\266\266\266\266\377\377\377\377\377\377\377\377\377\377\377\377\044\377"
                "\377\377\377\377\377\050\064\064\064\000\000\000\000\072\376\376\350\350\350\350\350\350\377\377\377\377\377\377\377\377\377\377\377\377\377\005\377\377\377\377"
                "\350\350\350\350\350\350\350\310\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\310\000\064\064\064\064\064\064\064\064\064\064\064\064\064\064\073\064\064\064\064"
                "\064\064\064\064\064\064\064\064\000\377\365\374\365\365\365\365\001\236\365\000\000\001\000\000\000\000\000\000\000\000\000\000\001\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\064\064\064\064\064\064\064\064\153\153\153\153\153"
                "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\337\266\266\266\000"
                "\000\000\000\000\000\000\377\377\377\377\377\000\000\000\000\000\000\000\100\377\377\350\350\000\000\350\350\350\350\350\350\350\350\001\000\000\000\000\000\000"
                "\036\103\012\072\103\365\000\000\000\000\000\000\000\000\000\377\377\377\350\350\350\350\350\350\350\350\350\350\350\350\350\350\254\350\350\350\350\354\350\350"
                "\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\221\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\050\000\050\050\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\004\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\000\177\377\377\377"
                "\046\000\000\000\000\000\000\027\027\027\027\027\027\027\027\027\027\027\027\000\027\027\027\027\000\004\000\000\000\000\000\135\000\044\103\000\000\377\377\377"
                "\377\377\103\377\364\377\364",
                2167
            }, /* #300 (#2) Andre Maute */
    /* 16*/ { 10, BARCODE_AZTEC, -1, -1, -1, 1, -1,
                "\105\105\000\000\000\000\000\000\000\000\077\012\377\377\377\072\376\376\350\350\350\350\350\250\350\350\350\350\354\350\350\350\350\350\001\000\000\000\000\000"
                "\000\036\103\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\002\222\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\012"
                "\377\377\377\072\376\376\350\350\350\350\350\250\350\350\350\350\354\350\350\350\000\000\000\000\000\000\000\033\000\036\103\012\072\103\103\000\100\116\000\000"
                "\000\000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\012\000\000\000\000\072\103\103\103\103\012\012\000\365\365\365\365\365\365\374\365\365\365\365\365\000\000\001\000\000\000\000\000\100\377\337\377"
                "\377\377\377\377\000\000\000\000\372\377\000\100\377\377\350\350\000\000\350\350\350\350\350\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\365\000"
                "\000\000\000\000\000\000\000\000\377\377\377\350\350\350\350\350\350\350\350\350\350\350\350\350\350\061\350\350\350\350\354\350\350\350\350\350\001\000\000\000"
                "\000\000\000\036\103\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\216\000\000\000\000\000\000\000\377\377\377\377\377\377\377\000\000"
                "\377\365\374\365\365\365\365\001\236\365\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\064\064\064\064\064\064\064\064\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153"
                "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\064\064\064\064\064\064\064\064\064\064\044\064\064\064\064\064\064\064\064\064"
                "\064\064\064\064\064\064\064\064\064\064\064\064\064\064\064\064\064\264\264\362\362\362\362\242\242\242\242\242\242\242\242\242\242\242\242\242\242\242\242\242"
                "\242\242\242\242\242\242\242\242\242\242\242\242\242\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\012\000\000\000\000\072\103\103\103\103\012\012\000\365\365\365\365\365"
                "\365\374\365\365\365\365\365\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\100\377\377\377\377\377\377\377\000\000\000\000"
                "\000\000\000\100\377\377\350\350\000\000\350\350\350\350\350\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\365\000\000\000\000\000\000\000\266\266"
                "\266\266\112\000\000\000\266\266\266\266\266\266\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\022\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\012\377"
                "\377\377\072\376\376\350\350\350\350\350\000\000\000\000\001\000\000\000\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\100\116\000\000\000"
                "\000\000\000\000\000\000\000\000\222\000\000\000\000\000\000\000\000\000\000\025\000\000\000\000\001\000\000\000\000\000\000\003\000\000\000\000\000\000\000\000"
                "\000\000\012\000\000\000\000\072\103\103\103\103\012\012\000\365\365\365\365\365\365\374\365\365\365\365\365\000\000\000\311\000\000\000\000\100\377\337\377\377"
                "\377\377\377\000\000\000\000\000\000\000\100\377\377\350\353\000\000\350\150\350\350\350\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\365\000\000"
                "\000\000\000\000\000\047\000\377\377\377\350\350\350\350\350\350\350\350\350\350\350\350\350\350\254\350\350\350\350\354\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\350\350\350\350\350\001\000\000\127\000\000\000\036\103"
                "\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\220\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\050\050\050\000\000\000"
                "\000\000\000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\000\000\000\000\000\000\000\000\000\000\000\000\266"
                "\266\266\266\266\266\377\377\377\377\377\013\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\005\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\172\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\266\266\266\266\266\266\266\266\266\266\377\377\377\377\377\377\377\377\377\377\377\377\044"
                "\377\377\377\377\377\377\050\064\064\064\000\000\000\000\072\376\376\350\350\350\350\350\350\377\377\377\377\377\377\377\377\377\377\377\377\377\005\377\377\377"
                "\377\350\350\350\350\350\350\350\310\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\310\000\064\064\064\064\064\064\064\064\064\064\064\064\064\064\073\064\064\064\064\064\064\064\064\064\064\064"
                "\064\000\377\365\374\365\365\365\365\001\236\365\000\000\001\000\000\000\000\000\000\000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\064\064\064\064\064\064\064\064\153\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\153\153\153\153"
                "\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\153\337\266\266\266\000"
                "\000\000\000\000\000\000\377\377\377\377\377\000\000\000\000\000\000\000\100\377\377\350\350\000\000\350\350\350\350\350\350\350\350\001\000\000\000\000\000\000"
                "\036\103\012\072\103\365\000\000\000\000\000\000\000\000\000\377\377\377\350\350\350\350\350\350\350\350\350\350\350\350\350\350\254\350\350\350\350\354\350\350"
                "\350\350\350\001\000\000\000\000\000\000\036\103\012\072\103\103\000\100\116\000\000\000\000\000\000\000\000\000\000\000\221\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\050\000\050\050\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\004\000\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032"
                "\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032"
                "\032\032\032\032\032\032\032\032\032\032\032\032\032\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\032\032\032\000\000\000\000\000\000\000\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\000\177\377\377\377\046\000\000\000\000\000\000\027\027\027"
                "\027\027\027\027\027\027\027\027\027\000\027\027\027\027\000\004\000\000\000\000\000\135\000\044\103\000\000\377\377\377\377\377\103\377\364\377\364",
                2157
            }, /* #300 (#3) Andre Maute */
    /* 17*/ { 11, BARCODE_AZTEC, -1, -1, 2, -1, -1,
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000",
                996
            }, /* Padding 11 example causing issue with ZXing-C++ */
    /* 18*/ { 12, BARCODE_AZTEC, -1, -1, -1, -1, -1,
                "\133\060\060\060\135\060\125\125\125\125\140\060\125\125\125\125\060\060\060\271\060\060\125\103\164\125\125\125\377\377\125\125\125\125\125\125\125\133\060\076"
                "\060\135\261\177\261\261\261\236\261\261\261\040\261\261\261\261\261\261\261\020\261\261\261\261\261\261\265\261\261\261\261\261\261\261\261\261\261\261\261\040"
                "\224\261\261\261\261\261\000\000\004\000\031\060\031\031\031\031\031\031\031\031\261\261\040\261\261\261\261\261\261\261\020\261\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\373\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\376\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\223\000\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                "\377\377\377\377\377\261\261\261\261\261\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\135\135\135\135\135\135\135\335\135\060\060\010\010\010\010"
                "\010\060",
                4402
            }, /* fuzz_data (1st) */
    /* 19*/ { 0, BARCODE_PDF417, FAST_MODE, -1, -1, -1, -1,
                "\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075"
                "\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\073\060\025\060\075\060\204\060\103\204\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060"
                "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                "\000\075\060\226\060\100\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\110\060\160\060\075\060\075\204\060"
                "\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\073\060\074\060\075"
                "\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\120\214\060\204\060\074\204\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041"
                "\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\120\214\060\204\060\075\060\204\060\075\204\060\204\060\204\075\060\075\060"
                "\204\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060"
                "\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\075\060\103\204\060\214\060\204\060\075\060"
                "\200\060\204\075\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060\075\204\060\204\060\204\075\060\075\060\204\060\134\060\075\204\060\040\060\103\204"
                "\060\372\120\060\124\060\004\060\103\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\000\060\113\060\377\060\235\060\075\060"
                "\204\060\103\204\060\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\211\060\214\060\204\060\075\060"
                "\041\060\204\060\204\060\120\060\075\060\204\060\003\060\103\204\060\204\060\377\060\350\060\223\060\000\075\060\226\060\103\204\060\204\060\204\120\060\075\060"
                "\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120"
                "\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\155\060\000\075\060\226\060\100\204\060\204\060"
                "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                "\000\075\060\226\060\100\204\060\204\060\204\075\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\075\060\110\060\160"
                "\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075"
                "\060\073\060\074\060\075\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\214\060\214"
                "\060\204\060\075\060\041\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\075\214\060\204\060\075\060\204\060\075\204\060\204"
                "\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103"
                "\204\060\214\060\204\060\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\004\060\103\204\060"
                "\204\060\003\060\204\075\060\120\214\060\204\060\004\060\103\204\060\204\060\003\060\211\074\060\120\060\124\060\351\060\120\060\075\060\351\060\072\375\060\204\060",
                1001
            }, /* Original OSS-Fuzz triggering data */
    /* 20*/ { 1, BARCODE_PDF417, -1, -1, -1, -1, -1,
                "\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075"
                "\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\073\060\025\060\075\060\204\060\103\204\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060"
                "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                "\000\075\060\226\060\100\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\110\060\160\060\075\060\075\204\060"
                "\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\073\060\074\060\075"
                "\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\120\214\060\204\060\074\204\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041"
                "\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\120\214\060\204\060\075\060\204\060\075\204\060\204\060\204\075\060\075\060"
                "\204\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060"
                "\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\075\060\103\204\060\214\060\204\060\075\060"
                "\200\060\204\075\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060\075\204\060\204\060\204\075\060\075\060\204\060\134\060\075\204\060\040\060\103\204"
                "\060\372\120\060\124\060\004\060\103\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\000\060\113\060\377\060\235\060\075\060"
                "\204\060\103\204\060\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\211\060\214\060\204\060\075\060"
                "\041\060\204\060\204\060\120\060\075\060\204\060\003\060\103\204\060\204\060\377\060\350\060\223\060\000\075\060\226\060\103\204\060\204\060\204\120\060\075\060"
                "\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120"
                "\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\155\060\000\075\060\226\060\100\204\060\204\060"
                "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                "\000\075\060\226\060\100\204\060\204\060\204\075\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\075\060\110\060\160"
                "\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075"
                "\060\073\060\074\060\075\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\214\060\214"
                "\060\204\060\075\060\041\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\075\214\060\204\060\075\060\204\060\075\204\060\204"
                "\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103"
                "\204\060\214\060\204\060\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\004\060\103\204\060"
                "\204\060\003\060\204\075\060\120\214\060\204\060\004\060\103\204\060\204\060\003\060\211\074\060\120\060\124\060\351\060\120\060\075\060\351\060\072\375\060\204\060",
                1001
            }, /* Original OSS-Fuzz triggering data */
    /* 21*/ { 28, BARCODE_PDF417COMP, FAST_MODE, -1, 0, -1, -1,
                "\000\000\000\377\377\010\002\000\000\033\005\031\000\000\002\000\000\000\000\101\101\101\101\101\101\101\101\000\000\000\000\000\000\000\374\000\101\101\101\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\323\000\000\000\000\000\010\000\000\000\000\165\000\000\000\000\000\000\000\000\000\000\000\000\056"
                "\000\000\000\000\000\000\000\000\000\000\100\000\000\101\101\101\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\323\000\000\000\000\000\010\000"
                "\000\000\000\000\000\000\000\000\000\000\000\323\000\000\000\000\000\010\000\000\000\000\165\000\000\000\000\000\000\000\000\000\000\000\000\056\000\000\000\000"
                "\000\000\000\000\000\000\000\000\101\101\101\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\323\000\010\000\000\000\010\000\000\000\000\165\000"
                "\000\000\000\000\000\000\000\000\000\000\000\056\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\375\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\040\000\000\000\000\000\000\000\000\000\000\101\101\101\101\101\101\101\101\101\101\101\055\101\101\101\101\101\101\101\101\101"
                "\101\101\101\060\151\003\000\000\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\137\101\101\101\101\101\101\101\101\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\101\101\101\101\101\101\101\101\101\101\000\000\000\000\000\000\000\000\000\070\000\000\000\000\000\000\000"
                "\000\000\000\377\377\010\002\000\000\033\005\031\000\000\002\000\000\000\000\101\101\101\101\101\101\101\101\000\000\000\000\000\000\000\374\000\101\101\101\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\323\000\000\000\000\000\010\000\000\000\000\165\000\000\000\000\000\000\000\000\000\000\000\000\056"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\375\000\000\000\000\000\000\000\000\000\000\000\000\000\040\000\000\000"
                "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\377\377\377\377\377\377\377\375\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\040\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\101\101\101\101\101\101\101\101\101\101\101\101\101\055\101\101\101\101\101\101\101\101\101"
                "\101\101\101\060\151\003\000\000\101\101\101\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\373\377\377\377\377\060"
                "\060\060\060\060\060\060\060\060\060\060\060\051\060\060\060\060\060\060\377\377\377\377\000\000\000\000\377\161\000\151\151\250\122\141\012\377\377\021\021\021"
                "\021\021\021\161\021\000\324\324\324\324\324\324\324\324\324\324\324\000\000\000\000\112\000\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324"
                "\320\324\324\324\324\021\176\012\000\000\000\000\000\000\324\324\324\324\324\324\324\101\101\101\101\101\101\101\352\352\352\352\352\352\352\352\352\352\352\352"
                "\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\352\000"
                "\000\000\000\000\101\101\101\101\101\101\101\101\101\101\101\101\101\137\101\101\101\101\101\101\101\101\000\000\000\000\000\000\000\000\000\000\000\000\101\101"
                "\101\101\041\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\324\324\324\324\324\324\324\324\324\077\324\324\324\324\324\324\324\324\324\324"
                "\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\324\000\060\060\060\060\060"
                "\060\060\060\060\060\060\060\060\060\060\060\051\060\060\060\060\060\060\377\377\377\377\000\000\000\000\377\161\000\151\151\250\122\141\012\377\377\021\021\021"
                "\021\021\021\161\021\000\324\324\324\324\324\324\324\324\324\324\324\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\101\137\101\101\101\101\101"
                "\101\000\000\000\000\374\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\323\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\323\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000",
                1048
            }, /* #300 (#1) Andre Maute (`pdf_text_submode_length()` not checking if previous/next BYT) */
    /* 22*/ { 29, BARCODE_PDF417, FAST_MODE, -1, -1, -1, -1,
                "\060\060\060\060\060\060\060\060\060\060\060\162\162\162\162\162\162\162\162\162\162\047\122\162\000\000\167\211\206\001\000\047\153\153\153\153\153\067\066\164"
                "\060\060\060\060\060\060\060\060\060\060\060\162\162\162\162\162\162\162\162\162\162\047\122\162\000\000\167\211\206\001\000\047\153\153\153\153\153\153\153\164"
                "\164\164\164\164\164\124\164\164\164\164\164\164\164\164\164\164\164\164\164\164\060\060\060\060\060\060\060\060\060\060\060\162\162\162\162\162\162\162\162\162"
                "\162\047\122\162\162\162\162\162\162\162\167\167\167\162\162\162\162\047\122\162\000\000\167\167\167\001\152\152\152\152\152\152\051\050\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\107\107\107\107\107\107\107\107"
                "\107\107\162\107\107\107\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\152\152\152\152\152"
                "\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152"
                "\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050"
                "\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051"
                "\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152"
                "\152\152\051\050\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051"
                "\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152"
                "\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051"
                "\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107"
                "\107\107\107\107\152\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051"
                "\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152"
                "\152\152\152\152\152\107\107\107\107\107\152\051\051\051\152\051\050\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\107\107\107\152"
                "\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152"
                "\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\050\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152"
                "\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\107\107\107\152\051"
                "\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051"
                "\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107"
                "\107\107\107\107\152\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\050\152"
                "\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152"
                "\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\050\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051"
                "\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051"
                "\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152"
                "\152\152\152\152\152\152\152\152\107\107\107\107\107\152\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\000\124\164\164\162\162\162\047\122\162\162"
                "\162\162\001\100\167\167\001\044\204\167\167",
                2611
            }, /* #300 (#7) Andre Maute */
    /* 23*/ { 30, BARCODE_PDF417, -1, -1, -1, -1, -1,
                "\060\060\060\060\060\060\060\060\060\060\060\162\162\162\162\162\162\162\162\162\162\047\122\162\000\000\167\211\206\001\000\047\153\153\153\153\153\067\066\164"
                "\060\060\060\060\060\060\060\060\060\060\060\162\162\162\162\162\162\162\162\162\162\047\122\162\000\000\167\211\206\001\000\047\153\153\153\153\153\153\153\164"
                "\164\164\164\164\164\124\164\164\164\164\164\164\164\164\164\164\164\164\164\164\060\060\060\060\060\060\060\060\060\060\060\162\162\162\162\162\162\162\162\162"
                "\162\047\122\162\162\162\162\162\162\162\167\167\167\162\162\162\162\047\122\162\000\000\167\167\167\001\152\152\152\152\152\152\051\050\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\107\107\107\107\107\107\107\107"
                "\107\107\162\107\107\107\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\152\152\152\152\152"
                "\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152"
                "\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050"
                "\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051"
                "\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152"
                "\152\152\051\050\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051"
                "\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152"
                "\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051"
                "\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107"
                "\107\107\107\107\152\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051"
                "\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152"
                "\152\152\152\152\152\107\107\107\107\107\152\051\051\051\152\051\050\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\107\107\107\152"
                "\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152"
                "\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\050\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152"
                "\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\107\107\107\152\051"
                "\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051"
                "\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107"
                "\107\107\107\107\152\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\051\051\051\050\152"
                "\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152"
                "\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\050\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051"
                "\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051"
                "\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152"
                "\152\152\152\152\152\152\152\152\107\107\107\107\107\152\051\051\051\051\107\107\107\107\107\107\107\107\107\107\107\051\051\051\051\051\051\050\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\051\152\152\152\152\152\152\152\152\152\152\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051"
                "\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\051\051\050\152\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\152\152\051\051\051\051\051\051\051\051\051"
                "\051\051\051\051\107\107\051\051\051\051\051\051\051\051\051\051\051\051\050\051\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051"
                "\051\051\050\152\152\152\152\152\152\152\152\051\050\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\051\000\124\164\164\162\162\162\047\122\162\162"
                "\162\162\001\100\167\167\001\044\204\167\167",
                2611
            }, /* #300 (#7) Andre Maute !FAST_MODE */
    /* 24*/ { 31, BARCODE_PDF417, FAST_MODE, -1, -1, 0 /*same effect as 242*/, -1,
                "\000\000\000\000\000\000\000\000\000\000\000\212\377\000\000\153\153\153\153\153\153\153\060\047\047\043\047\057\153\000\153\153\137\377\153\153\000\134\000\000"
                "\000\153\153\343\153\153\153\060\047\047\043\047\057\157\153\153\153\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\377\377\377\167"
                "\001\100\000\377\004\002\000\000\000\000\000\001\000\000\134\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162"
                "\077\162\072\176\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\364\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362"
                "\134\162\162\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077"
                "\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\377\134\134\126\142\134"
                "\162\362\162\162\162\072\176\000\162\134\134\126\142\134\162\362\162\162\162\072\176\000\162\162\162\162\162\174\174\377\134\362\162\362\162\162\162\072\176\000"
                "\162\362\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\000\044\162\162\162\162\174\174\377\134\362\162\162\162\134\134\134\142\162"
                "\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000"
                "\162\162\377\377\377\377\377\377\134\134\134\142\134\162\153\153\153\153\153\153\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134"
                "\134\134\142\134\162\362\162\162\162\072\176\000\162\000\001\000\000\134\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162"
                "\162\162\162\077\162\072\176\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\364\072\176\000\162\162\162\162\162\174\174\377\134\134\134"
                "\162\142\362\134\162\162\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162"
                "\162\162\077\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\162\162\072"
                "\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\377\134\134\126\142\134\162\362\162\162\162\072\176\000\174\174\377\134\362\162\362\162\162\162"
                "\072\176\000\162\362\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\162\362\162\162\162\134\134\134\142\134\162\162\162\162\077\173"
                "\153\153\153\165\000\000\000\153\151\153\153\153\153\153\153\153\153\153\153\153\153\153\153\176\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362"
                "\162\162\162\072\176\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362"
                "\162\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\377\134\134\134\167\167\167\167\167\001\100\000\002\000\000\000\000\000\000\000"
                "\000\153\153\067\000\000\000\153\153\300\000\000\000\000\000\000\000\000\000\000\000\000\212\377\000\000\153\153\153\153\153\153\153\060\047\047\043\047\057\153"
                "\000\153\153\137\377\153\153\000\134\000\000\000\153\153\343\153\153\153\060\047\047\043\047\057\157\153\153\153\162\162\077\162\072\176\000\162\134\142\134\162"
                "\162\162\162\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\377\377\377\167\001\100\000\377\004\002\000\000\000\000"
                "\000\001\000\000\134\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\362\377\377\377"
                "\377\377\377\134\134\134\142\134\162\362\162\162\364\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\215\215\162"
                "\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377"
                "\377\377\134\134\134\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\377\134\134\126\142\134\162\362\162\162\162\072\176\000\162\162"
                "\162\162\162\174\174\377\134\362\162\362\162\162\162\072\176\000\162\362\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\000\044\162"
                "\162\162\162\174\174\377\134\362\162\162\162\134\134\134\142\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377"
                "\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\153\153\153\153\153\153\142\134\162\162"
                "\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\000\001\000\000\134\077\162\072\176\000\162"
                "\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162"
                "\162\364\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162"
                "\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\162\162\072\176\000\162"
                "\162\162\162\162\174\174\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\162\162\162\162\174\174\377\134\362\162\172\162\134\134\134\142\162\072\162"
                "\162\162\162\174\174\377\134\362\162\362\162\162\162\072\176\000\162\362\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\000\044\162"
                "\162\162\162\174\174\377\134\362\162\162\162\134\134\134\142\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377"
                "\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\153\153\153\153\153\153\142\134\162\162"
                "\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\000\001\000\000\134\077\162\072\176\000\162"
                "\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162"
                "\162\364\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162"
                "\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072"
                "\176\000\162\162\377\377\377\377\377\377\134\134\134\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\377\134\134\126\142\134\162\362"
                "\162\162\162\072\176\000\174\174\377\134\362\162\362\162\162\162\072\176\000\162\362\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000"
                "\162\362\162\162\162\134\134\134\142\134\162\162\162\162\077\173\153\153\153\165\000\000\000\153\151\153\153\153\153\153\153\153\153\153\153\153\153\153\153\176"
                "\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\162\072\176\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176"
                "\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\377\134\134\134"
                "\167\167\167\167\167\001\100\000\002\000\000\000\000\000\000\000\000\153\153\067\000\000\000\153\153\300\000\000\000\000\000\000\000\000\000\000\000\000\212\377"
                "\000\000\153\153\153\153\153\153\153\060\047\047\043\047\057\153\000\153\153\137\377\153\153\000\134\000\000\000\153\153\343\153\153\153\060\047\047\043\047\057"
                "\157\153\153\153\162\162\077\162\072\176\000\162\134\142\134\162\162\162\162\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134"
                "\162\162\377\377\377\167\001\100\000\377\004\002\000\000\000\000\000\001\000\000\134\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134"
                "\142\134\162\162\162\162\077\162\072\176\000\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\364\072\176\000\162\162\162\162\162\174\174\377"
                "\134\134\134\162\142\362\134\162\162\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142"
                "\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362"
                "\377\134\134\126\142\134\162\362\162\162\162\072\176\000\162\162\162\162\162\174\174\377\134\362\162\362\162\162\162\072\176\000\162\362\162\162\162\174\174\377"
                "\134\134\134\162\142\362\134\162\162\162\072\176\000\000\044\162\162\162\162\174\174\377\134\362\162\162\162\134\134\134\142\162\072\176\000\215\215\162\162\162"
                "\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377"
                "\134\134\134\142\134\162\153\153\153\153\153\153\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162"
                "\162\072\176\000\162\000\001\000\000\134\077\162\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000"
                "\162\362\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\364\072\176\000\162\162\162\162\162\174\174\377\134\134\134\162\142\362\134\162\162\162\072"
                "\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162"
                "\162\377\377\377\377\377\377\134\134\134\162\162\072\176\000\162\162\162\162\162\174\174\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\162\162\162"
                "\162\174\174\377\134\362\162\172\162\134\134\134\142\162\072\176\000\215\215\162\162\162\174\174\174\171\072\176\000\162\162\162\162\162\162\377\053\377\377\377"
                "\134\134\134\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\362\162\162\162"
                "\174\174\377\134\134\134\162\142\362\134\162\162\162\072\176\000\162\362\162\162\162\134\134\142\134\162\362\162\162\162\072\176\000\044\162\162\162\162\174\174"
                "\377\134\362\162\162\162\134\134\134\142\162\072\176\000\215\215\162\162\162\174\174\174\174\072\176\000\162\162\162\162\162\162\377\053\377\377\377\134\134\134"
                "\142\134\162\162\162\162\077\162\072\176\000\162\162\377\377\377\377\377\377\134\134\134\142\134\162\362\162\162\162\072\176\000\162\162\162\162\162\174\174\377"
                "\134\134\134\162\142\362\134\162\162\162\072\176\000\215\215\162\162\162\174\174\174\174\134\134\134\142\134\000\153\153\153\153\153\153\153\062\047\047\043\047"
                "\057\262\054\377\134\134\142\153\330\153",
                2690
            }, /* #300 (#10) Andre Maute */
    /* 25*/ { 1, BARCODE_QRCODE, -1, -1, -1, -1, ZINT_FULL_MULTIBYTE, "\215\215\350\2156750\215\215\215\215\215\215\000\000\000\025\215\215\215\215\215\232\215\232\232\001\361\215\215\215\215\215\221\215\215\215\215JJJJJJNJJJJJJ\215\215\215\2159999\215\215\215\215\215\215\215\215\215\235\215\215\215\215\215\035\004\000\000@\000\000\000\000\375\000\000\000\000\000\000\000\000\000\000\000\000\000\241\000\000\000\000\000\000\000\241\247^^^\377\377\377\000 \000\000\000\000\000\000\377\377u\000\000\000\000\000\000\000^\377\377^^\000:\000\177\377\377\377?\377\377\377\377\377\377\377\377\377\377\377\377\377\377\241\241\232\232\232\232\232\232\232\232\000\377\377\377\242\003\000\000\377\377/\000AA\000\000\000\000\000\000\000\000\000\000\000\000T\000\000\000\000\000\000\000\000WWW\237\250WWWWWW\377\377R30      \377\377\000\000\000", 231 }, /* #300 (#15), Andre Maute */
    /* 26*/ { 2, BARCODE_QRCODE, -1, 35, -1, -1, ZINT_FULL_MULTIBYTE, "\215\215\215\215\215\350\215\215999\215\21500000\215\215\215\215\215\215\377O\000\000\036\000\000\000\000\357\376\026\377\377\377\377\241\241\232\232\232\232\232\232\235\032@\374:JGB \000\000@d\000\000\000\241\241\000\000\027\002\241\241\000\000\014\000\000\000\000\357\327\004\000\000\000\000\000\000\000\375\000\000\000\000\000\000\000\000\000\000\000\000\0000253]9R4R44,44,4404[255\350999\215\21599999\215\215\215\2150000\215\215\215\215\215\215\215\215\215]9444442<4444,4044%44vA\000\000\002\000'\000\000\215\377@\215\215\350\215\215\215\215\215\215\215\307\306\306n\215\215\000\000\001\000\000\203\000\000\000\000\000\000@\215\215\215[\2154315@]R0", 229 }, /* #300 (#16), Andre Maute */
    /* 27*/ { 0, BARCODE_AUSROUTE, -1, -1, -1, -1, -1, "A\000\000\000", 4 }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 28*/ { 1, BARCODE_AUSROUTE, -1, -1, -1, -1, -1, "1\000\000\000", 4 }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 29*/ { 0, BARCODE_EANX, -1, -1, -1, -1, -1, "55++15", -1, }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 30*/ { 1, BARCODE_EANX, -1, -1, -1, -1, -1, "+123456789012345678", -1, }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 31*/ { 8, BARCODE_EANX, -1, -1, -1, -1, -1, "+12345", -1, }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 32*/ { 9, BARCODE_EANX, -1, -1, -1, -1, -1, "+123456", -1, }, /* #181 Christian Hartlage OSS-Fuzz */
    /* 33*/ { 10, BARCODE_EANX, -1, -1, -1, -1, -1, "000002000000200+203", -1 }, /* #218 Jan Schrewe CI-Fuzz */
    /* 34*/ { 0, BARCODE_DOTCODE, -1, -1, -1, -1, -1, "(\207'", -1 }, /* #181 Christian Hartlage / Nico Gunkel OSS-Fuzz */
    /* 35*/ { 1, BARCODE_DOTCODE, -1, -1, -1, -1, -1,
                "\133\061\106\133\061\106\070\161\116\133\116\116\067\040\116\016\000\116\125\111\125\125\316\125\125\116\116\116\116\117\116\125"
                "\111\125\103\316\125\125\116\116\116\116\117\000\000\116\136\116\116\001\116\316\076\116\116\057\136\116\116\134\000\000\116\116"
                "\116\230\116\116\116\116\125\125\125\257\257\257\000\001\116\130\212\212\212\212\212\212\212\377\377\210\212\212\177\000\212\212"
                "\212\212\212\212\175\212\212\212\212\212\212\116\117\001\116\116\112\116\116\116\116\176\136\000\000\000\000\000\000\000\000\000"
                "\000\000\000\000\000\000\000\000\005\377\377\005\125\125\125\325\001\116\116\116\266\116\020\000\200\000\116\116\177\000\000\377"
                "\377\257\257\257\125\112\117\116\001\000\000\044\241\001\116\116\116\136\116\116\116\056\116\125\111\125\125\316\125\125\116\116"
                "\116\116\057\000\000\116\136\116\116\001\116\116\076\342\116\057\136\116\116\134\000\000\116\116\116\241\116\116\116\116\125\125"
                "\125\257\257\257\000\001\116\130\212\212\212\212\212\212\212\212\172\212\071\071\071\071\071\071\071\071\071\071\071\071\071\071"
                "\071\071\071\071\071\110\071\071\051\071\065\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071"
                "\071\071\071\071\071\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330\330"
                "\330\330\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\071\065\071\071\071\071\071\071\071\071\071\071\071\071\071"
                "\071\071\071\071\071\072\071\071\277\071\071\077\071\071\071\071\071\071\071\071\154\071\071\071\071\071\071\071\071\071\071\071"
                "\071\071\071\011\071\071\071\071\071\071\071\071\071\071\071\071\071\071\105\105\105\105\105\105\105\105\105\105\105\105\105\071"
                "\071\071\071\071\071",
                421
            }, /* Original OSS-Fuzz triggering data for index out of bounds (encoding of HT/FS/GS/RS when shifting to code set B) */
    /* 36*/ { 2, BARCODE_DOTCODE, -1, -1, -1, -1, -1, "\233:", -1 }, /* Original OSS-Fuzz triggering data for codeword_array buffer overflow, L777 */
    /* 37*/ { 3, BARCODE_DOTCODE, -1, -1, -1, -1, -1, "\241\034", -1 }, /* As above L793 */
    /* 38*/ { 4, BARCODE_DOTCODE, -1, -1, -1, -1, -1, "\270\036", -1 }, /* As above L799 */
    /* 39*/ { 5, BARCODE_DOTCODE, -1, -1, -1, -1, -1, "\237\032", -1 }, /* As above L904 */
    /* 40*/ { 6, BARCODE_DOTCODE, -1, -1, -1, -1, -1, "\237", -1 }, /* As above L1090 */
    /* 41*/ { 0, BARCODE_MAXICODE, -1, -1, -1, -1, -1, "\223\223\223\223\223\200\000\060\060\020\122\104\060\343\000\000\040\104\104\104\104\177\377\040\000\324\336\000\000\000\000\104\060\060\060\060\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\104\060\104\104\000\000\000\040\104\104\104\104\177\377\377\377\324\336\000\000\000\000\104\377\104\001\104\104\104\104\104\104\233\233\060\060\060\060\060\060\060\060\060\325\074", 107 }, /* #181 Nico Gunkel OSS-Fuzz - original OSS-Fuzz triggering data */
    /* 42*/ { 1, BARCODE_MAXICODE, -1, -1, -1, -1, -1, "AaAaAaAaAaAaAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA123456789", -1 }, /* Add 6 lowercase a's so 6 SHIFTS inserted so 6 + 138 (max input len) = 144 and numbers come at end of buffer */
};

/* GS1_MODE data */
static const struct item gs1_data[] = {
    /*  0*/ { 0, BARCODE_QRCODE, GS1_MODE | GS1NOCHECK_MODE, -1, -1, -1, -1, "[]CCCCCLLLLLLLLLLLLLLLLLLLLLCCCCCCCC@CCCCCCCCCCCCCCCCCCCCCCC%%C%C%%%%%%%%%%%%%%LLLCCCCCCCC%%C%C%%%%%%%%%%%%%%LLLLLLLLLLLLLLLLLLL000000032861710*383556LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL155816162LLLLLCC%%C%C%%%%%%%%%%%%%%LLLCCCCCCCC%%C%C%%%%%%%%%%%%%%LLLLLLLLLL)!1661055777[723]T5", -1 }, /* #300 (#14), Andre Maute */
    /*  1*/ { 5, BARCODE_EANX_CC, GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, -1, -1, -1, "()111%", -1 }, /* #300 (#5), Andre Maute (`dbar_date()` not checking length + other non-checks) */
    /*  2*/ { 6, BARCODE_UPCA_CC, GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, -1, -1, -1, "()90", -1, }, /* #300 (#6), Andre Maute (`dbar_date()` not checking length + other non-checks) */
    /*  3*/ { 7, BARCODE_UPCA_CC, GS1_MODE | GS1PARENS_MODE | GS1NOCHECK_MODE, -1, -1, -1, -1, "()904OOOOO)CK0336680OOOOOOOOOOOOOO29[0kkkk%%%%(", -1 }, /* #300 (#11), Andre Maute (`gs1_verify()` not checking length on resolve AI data loop) */
    /*  4*/ { 8, BARCODE_GS1_128_CC, GS1_MODE | GS1NOCHECK_MODE, -1, 3, -1, -1, "[]RRR___________________KKKRRR0000", -1 }, /* #300 (#13), Andre Maute (`calc_padding_ccc()` dividing by zero when linear width == 68) */
};

/* Write a setting as 1 char to filename, allowing for -1 meaning none (255 or zero) */
static void write_setting(const char *const filename, const int setting, const unsigned char none, FILE *fp) {
    unsigned char ch = setting == -1 ? none : (unsigned char) setting;
    size_t ret = fwrite(&ch, sizeof(unsigned char), 1, fp);
    if (ret != 1) {
        fprintf(stderr, "fwrite(%s) failed ret %d != 1 (%d: %s)\n", filename, (int) ret, errno, strerror(errno));
        assert(0);
    }
}

/* Creates directory `dirname` if not already existing, and writes each `item` of `data` to files named
   "dirname/NNNN_BARCODE_NAME" */
static void write_corp(const char dirname[], const struct item data[], const int data_size, const int no_eci) {
    struct stat st;
    int i;
    char filename[2048];
    FILE *fp;

    if (mkdir(dirname, S_IRWXU) != 0 && errno != EEXIST) {
        fprintf(stderr, "mkdir(%s) failed (%d: %s)\n", dirname, errno, strerror(errno));
        assert(0);
    }
    if (stat(dirname, &st) != 0 || !S_ISDIR(st.st_mode)) {
        fprintf(stderr, "%s not directory\n", dirname);
        assert(0);
    }

    for (i = 0; i < data_size; i++) {
        const size_t length = data[i].length == -1 ? strlen(data[i].data) : data[i].length;
        char name[32] = {0};
        const struct settings_item *si;
        size_t ret;
        char ch;

        if (ZBarcode_BarcodeName(data[i].symbology, name) != 0) {
            assert(0);
        }
        sprintf(filename, "%s/%04d-%02d_%s", dirname, i, data[i].test_fuzz_idx, name + 8); /* Lose "BARCODE_" */
        fp = fopen(filename, "w+");
        if (!fp) {
            fprintf(stderr, "fopen(%s, w+) failed (%d: %s)\n", filename, errno, strerror(errno));
            assert(fp);
        }

        assert(data[i].symbology >= 0 && data[i].symbology < ZARRAY_SIZE(settings));
        si = settings + data[i].symbology;

        if (data[i].input_mode != -1) {
            /* Shift out DATA/UNICODE/GS1_MODE (shifted back in `set_symbol()` in "fuzz.h") */
            write_setting(filename, data[i].input_mode >> 3, 0, fp);
        } else {
            write_setting(filename, data[i].input_mode, 0, fp);
        }
        if (si->option_1_min <= si->option_1_max) { 
            write_setting(filename, data[i].option_1, 0xFF, fp);
        }
        if (si->option_2_min <= si->option_2_max) { 
            write_setting(filename, data[i].option_2, 0, fp);
        }
        if (!no_eci && (ZBarcode_Cap(data[i].symbology, ZINT_CAP_ECI) & ZINT_CAP_ECI)) {
            write_setting(filename, data[i].eci, 0, fp);
        }
        if (si->option_3) {
            write_setting(filename, data[i].option_3 != -1, 0, fp);
        }

        ret = fwrite(data[i].data, sizeof(char), length, fp);
        if (ret != length) {
            fprintf(stderr, "fwrite(%s) failed ret %d != length %d (%d: %s)\n",
                    filename, (int) ret, (int) length, errno, strerror(errno));
            assert(0);
        }
        fclose(fp);
    }
}

/* Write out DATA_MODE files and GS1_MODE files separately, in directories "fuzz_data_corpus" and "fuzz_gs1_corpus" */
int main(int argc, char **argv) {
    (void)argc; (void)argv;

    write_corp("fuzz_data_corpus", data_data, ZARRAY_SIZE(data_data), 0 /*no_eci*/);
    write_corp("fuzz_gs1_corpus", gs1_data, ZARRAY_SIZE(gs1_data), 1 /*no_eci*/);

    return 0;
}

#ifdef __cplusplus
}
#endif /* __cplusplus */

/* vim: set ts=4 sw=4 et : */
