# -*- coding: utf-8 -*-
# :-
# :Copyright: (c) 2018-2020, Franz Glasner. All rights reserved.
# :License:   3-clause BSD. See LICENSE.txt for details.
# :-
"""Read JSON-style configuration files.

"""

from __future__ import division, absolute_import, print_function


__all__ = ["load"]


import io
import json.decoder
try:
    from collections import OrderedDict as DictImpl
except ImportError:
    try:
        from ordereddict import OrderedDict as DictImpl
    except ImportError:
        DictImpl = dict

from .compat import u2fs


#
# Determine whether the JSONDecoder has the "object_pairs_hook"
# parameter once
#
try:
    json.decoder.JSONDecoder(object_pairs_hook=DictImpl)
except TypeError:
    _with_object_pairs_hook = False
else:
    _with_object_pairs_hook = True


def load(filename, encoding="utf-8"):
    """Load a single JSON file with name `filename` and encoding `encoding`.

    """
    with io.open(u2fs(filename), mode="rt", encoding=encoding) as jsfp:
        #
        # The scanner (not to be changed yet) does only recognize decimal
        # integers yet.
        #
        kwds = {
            "parse_int": lambda s: int(s, 0),
            "strict": False
        }
        if _with_object_pairs_hook:
            kwds["object_pairs_hook"] = DictImpl
        decoder = json.decoder.JSONDecoder(**kwds)
        data = decoder.decode(jsfp.read())
        if not isinstance(data, DictImpl):
            raise TypeError("JSON root must be an object (i.e. dict)")
        return data
