# -*- coding: utf-8 -*-
# :-
# :Copyright: (c) 2015-2022, Franz Glasner. All rights reserved.
# :License:   BSD-3-Clause. See LICENSE.txt for details.
# :-
"""Read TOML style configuration files.

"""

from __future__ import division, absolute_import, print_function


__all__ = ["load"]


import io
try:
    from collections import OrderedDict as DictImpl
except ImportError:
    try:
        from ordereddict import OrderedDict as DictImpl
    except ImportError:
        DictImpl = dict

try:
    import tomllib
except ImportError:
    tomllib = None
    try:
        import tomli
    except ImportError:
        tomli = None
        try:
            import toml
        except ImportError:
            toml = None
    else:
        toml = None
else:
    toml = tomli = None

from .compat import u2fs


def load(filename, encoding="utf-8"):
    """Load a single TOML file with name `filename` and encoding `encoding`.

    .. note:: The TOML standard requires that all TOML files are UTF-8
              encoded.

    """
    if tomllib:
        with open(u2fs(filename), mode="rb") as tfp:
            return tomllib.load(tfp)
    elif tomli:
        with open(u2fs(filename), mode="rb") as tfp:
            return tomli.load(tfp)
    elif toml:
        with io.open(
                u2fs(filename),
                mode="rt",
                encoding=encoding,
                newline="") as tfp:
            return toml.loads(tfp.read(), _dict=DictImpl)
    else:
        assert False
