# -*- coding: utf-8 -*-
r"""
configmix
^^^^^^^^^

A library for helping with configuration files.

:Author:    Franz Glasner
:Copyright: Copyright (c) 2015-2018, Franz Glasner.
            All rights reserved.
:License:   BSD 3-Clause License.
            See LICENSE.txt for details.
:ID:        $Header$

"""

from __future__ import division, print_function, absolute_import


__version__ = "0.5"

__revision__ = "$Revision$"


import copy

from .config import Configuration


__all__ = ["load", "Configuration"]


def load(*files):
    """Load the given configuration files, merge them in the given order
    and return the resulting :class:`configmix.config.Configuration` dictionary.

    """
    if not files:
        return Configuration()
    else:
        ex = merge(None, _load_cfg_from_file(files[0]))
        for f in files:
            ex = merge(_load_cfg_from_file(f), ex)
        return Configuration(ex)


def _load_cfg_from_file(filename):
    fnl = filename.lower()
    if fnl.endswith(".yml") or fnl.endswith("yaml"):
        from . import yaml
        with open(filename, "rb") as yf:
            return yaml.safe_load(yf)
    elif fnl.endswith(".py"):
        from . import py
        return py.load(filename)
    elif fnl.endswith(".ini"):
        from . import ini
        return ini.load(filename)
    else:
        raise ValueError("Unknown configuration file type for filename "
                         "%r" % filename)


if 0:
    #
    # From: https://github.com/jet9/python-yconfig/blob/master/yconfig.py
    # License: BSD License
    #
    def dict_merge(a, b):
        """Recursively merges dict's. not just simple a['key'] = b['key'], if
        both a and bhave a key who's value is a dict then dict_merge is called
        on both values and the result stored in the returned dictionary."""

        if not isinstance(b, dict):
            return b
        result = deepcopy(a)
        for k, v in b.iteritems():
            if k in result and isinstance(result[k], dict):
                result[k] = dict_merge(result[k], v)
            else:
                result[k] = deepcopy(v)
        return result


def merge(user, default, _first=True):
    """A simple (YAML-)tree merge.

    From http://stackoverflow.com/questions/823196/yaml-merge-in-python

    .. note:: `_first` is an internal argument.

    """
    if _first and (user is None):
        return default
    if isinstance(user, dict) and isinstance(default, dict):
        for k, v in default.items():
            if k not in user:
                user[k] = v
            else:
                user[k] = merge(user[k], v, False)
    return user


def safe_merge(user, default, _first=True):
    """A more safe version of :func:`merge()` that makes shallow copies of
    the returned container objects.

    .. note:: `_first` is an internal argument.

    """
    if _first and (user is None):
        return copy.copy(default)
    user = copy.copy(user)
    if isinstance(user, dict) and isinstance(default, dict):
        for k, v in default.items():
            if k not in user:
                user[k] = copy.copy(v)
            else:
                user[k] = merge(user[k], v, False)
    return user
