# -*- coding: utf-8 -*-
r"""Unit tests

"""

from __future__ import absolute_import, print_function

import io
import os
import sys
import unittest
try:
    from StringIO import StringIO
except ImportError:
    StringIO = None

from _test_setup import DATADIR

from cutils import shasum


PY2 = sys.version_info[0] <= 2


def _memfile():
    if StringIO:
        return StringIO()
    else:
        return io.StringIO()


class ChangedDir(object):

    """Context manager to temporarily change the directory"""

    def __init__(self, path):
        self._new_dir = path
        self._prev_dir = None

    def __enter__(self):
        self._prev_dir = os.getcwd()
        os.chdir(self._new_dir)

    def __exit__(self, *args, **kwds):
        if self._prev_dir is not None:
            os.chdir(self._prev_dir)
            self._prev_dir = None


class SignifyTests(unittest.TestCase):

    def test_empty(self):
        destfile = _memfile()
        opts = shasum.gen_opts(algorithm="SHA256",
                               dest=destfile,
                               files=[os.path.join(DATADIR, "empty")])
        shasum.shasum(opts)
        self.assertTrue(
            destfile.getvalue().startswith(
                "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855 *"))

    def test_empty_with_name(self):
        destfile = _memfile()
        with ChangedDir(DATADIR):
            opts = shasum.gen_opts(algorithm="SHA256",
                                   dest=destfile,
                                   files=["empty"])
            shasum.shasum(opts)
            self.assertEqual(
                "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855 *empty\n",
                destfile.getvalue())

    def test_empty_with_name_bsd(self):
        destfile = _memfile()
        with ChangedDir(DATADIR):
            opts = shasum.gen_opts(algorithm="SHA512",
                                   dest=destfile,
                                   files=["empty"],
                                   bsd=True)
            shasum.shasum(opts)
            self.assertEqual(
                "SHA512 (empty) = cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e\n",
                destfile.getvalue())

    def test_empty_mmap(self):
        destfile = _memfile()
        opts = shasum.gen_opts(algorithm="SHA256",
                               dest=destfile,
                               files=[os.path.join(DATADIR, "empty")],
                               mmap=True)
        shasum.shasum(opts)
        self.assertTrue(
            destfile.getvalue().startswith(
                "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855 *"))

    def test_empty_no_mmap(self):
        destfile = _memfile()
        opts = shasum.gen_opts(algorithm="SHA256",
                               dest=destfile,
                               files=[os.path.join(DATADIR, "empty")],
                               mmap=False)
        shasum.shasum(opts)
        self.assertTrue(
            destfile.getvalue().startswith(
                "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855 *"))


if __name__ == "__main__":
    sys.exit(unittest.main())
