# -*- coding: utf-8 -*-
# :-
# SPDX-FileCopyrightText: © 2025 Franz Glasner
# SPDX-License-Identifier: BSD-3-Clause
# :-
r"""Unit tests for treesum

"""

from __future__ import absolute_import, print_function

import os
import shutil
import sys
import unittest

from _test_setup import (DATADIR, TMPDIR)

import cutils.treesum


class TaggedTreesumTests(unittest.TestCase):

    def setUp(self):
        if not os.path.isdir(TMPDIR):
            os.mkdir(TMPDIR)

    def tearDown(self):
        if os.path.isdir(TMPDIR):
            shutil.rmtree(TMPDIR)

    def test_gen_and_info_P_utf8(self):
        digest_file = os.path.join(
            TMPDIR,
            "%d__test_gen_and_info_P_utf8.info" % (os.getpid(),))
        rel_tmpdir = os.path.relpath(TMPDIR, DATADIR)
        gen_opts = cutils.treesum.gen_generate_opts(
            directories=[DATADIR],
            algorithm="SHA1",
            fnmatch_filters=[("exclude", "glob:%s" % (rel_tmpdir,))],
            generator="full",
            grouping_separator="_",
            output=digest_file,
            output_style="tagged",
            print_size=True,
            utf8=True)
        info_opts = cutils.treesum.gen_info_opts(digest_files=[digest_file],
                                                 last=True)
        cutils.treesum.generate_treesum(gen_opts)
        cutils.treesum.print_treesum_digestfile_infos(info_opts)
        info = cutils.treesum.TreesumInfo.collect_last_from_file(digest_file)
        self.assertEqual(
            b"\xe1\x76\x18\xfc\x9b\x2f\x1e\x41\xfc\xe0\x04\xfc\x92\x43\x4e\xa7\x49\xbd\x57\x36",    # noqa: E501 line too long
            info.digest)
        self.assertEqual(2808, info.size)
        self.assertEqual("SHA1", info.algorithm)

    def test_gen_and_info_P_native(self):
        digest_file = os.path.join(
            TMPDIR,
            "%d__test_gen_and_info_P_native.info" % (os.getpid(),))
        rel_tmpdir = os.path.relpath(TMPDIR, DATADIR)
        gen_opts = cutils.treesum.gen_generate_opts(
            directories=[DATADIR],
            algorithm="SHA1",
            fnmatch_filters=[("exclude", "glob:%s" % (rel_tmpdir,))],
            generator="full",
            grouping_separator="_",
            output=digest_file,
            output_style="tagged",
            print_size=True,
            utf8=False)
        info_opts = cutils.treesum.gen_info_opts(digest_files=[digest_file],
                                                 last=True)
        cutils.treesum.generate_treesum(gen_opts)
        cutils.treesum.print_treesum_digestfile_infos(info_opts)
        info = cutils.treesum.TreesumInfo.collect_last_from_file(digest_file)
        self.assertEqual(
            b"\xe1\x76\x18\xfc\x9b\x2f\x1e\x41\xfc\xe0\x04\xfc\x92\x43\x4e\xa7\x49\xbd\x57\x36",    # noqa: E501 line too long
            info.digest)
        self.assertEqual(2808, info.size)
        self.assertEqual("SHA1", info.algorithm)

    def test_gen_and_info_L_utf8(self):
        digest_file = os.path.join(
            TMPDIR,
            "%d__test_gen_and_info_L_utf8.info" % (os.getpid(),))
        rel_tmpdir = os.path.relpath(TMPDIR, DATADIR)
        gen_opts = cutils.treesum.gen_generate_opts(
            directories=[DATADIR],
            algorithm="SHA1",
            fnmatch_filters=[("exclude", "glob:%s" % (rel_tmpdir,))],
            follow_symlinks=cutils.treesum.FollowSymlinkConfig(
                True, True, True),
            generator="full",
            grouping_separator="_",
            output=digest_file,
            output_style="tagged",
            print_size=True,
            utf8=True)
        info_opts = cutils.treesum.gen_info_opts(digest_files=[digest_file],
                                                 last=True)
        cutils.treesum.generate_treesum(gen_opts)
        cutils.treesum.print_treesum_digestfile_infos(info_opts)
        info = cutils.treesum.TreesumInfo.collect_last_from_file(digest_file)
        self.assertEqual(
            b"\xb3\xdd\x1b\x93\xec\x3a\xa5\xf1\xb1\xc6\xbd\x8c\x1e\xe1\xd5\xcf\x72\x65\x27\x1d",    # noqa: E501 line too long
            info.digest)
        self.assertEqual(2863, info.size)
        self.assertEqual("SHA1", info.algorithm)

    def test_gen_and_info_L_native(self):
        digest_file = os.path.join(
            TMPDIR,
            "%d__test_gen_and_info_L_native.info" % (os.getpid(),))
        rel_tmpdir = os.path.relpath(TMPDIR, DATADIR)
        gen_opts = cutils.treesum.gen_generate_opts(
            directories=[DATADIR],
            algorithm="SHA1",
            fnmatch_filters=[("exclude", "glob:%s" % (rel_tmpdir,))],
            follow_symlinks=cutils.treesum.FollowSymlinkConfig(
                True, True, True),
            generator="full",
            grouping_separator="_",
            output=digest_file,
            output_style="tagged",
            print_size=True,
            utf8=False)
        info_opts = cutils.treesum.gen_info_opts(digest_files=[digest_file],
                                                 last=True)
        cutils.treesum.generate_treesum(gen_opts)
        cutils.treesum.print_treesum_digestfile_infos(info_opts)
        info = cutils.treesum.TreesumInfo.collect_last_from_file(digest_file)
        self.assertEqual(
            b"\xb3\xdd\x1b\x93\xec\x3a\xa5\xf1\xb1\xc6\xbd\x8c\x1e\xe1\xd5\xcf\x72\x65\x27\x1d",    # noqa: E501 line too long
            info.digest)
        self.assertEqual(2863, info.size)
        self.assertEqual("SHA1", info.algorithm)


if __name__ == "__main__":
    sys.exit(unittest.main(buffer=True))
