#!/bin/sh
#
# Test helpers for the shell unittests using cram.
#


#:
#: Set some directories to temporary values for inclusion of test configuration
#: files.
#:
CONFIGDIR="${TESTDIR}/etc"
PACKAGE_MAPPING="${CONFIGDIR}/package-mapping.conf"


#:
#: Check that no global variables that hold any array storage are left.
#:
#: Returns:
#:   int: 0 if no unexpected storage is left, 1 otherwise
#:
check_no_array_artifacts() {
    # _farr_A_ is the storage prefix for arrays
    if set | grep -E -e '^_farr_A_.*='; then
	return 1
    else
	return 0
    fi
}


#:
#: Check that no global variables that hold any alist storage are left.
#:
#: Returns:
#:   int: 0 if no unexpected storage is left, 1 otherwise
#:
check_no_alist_artifacts() {
    # This are all _farr_alist_XXX_prefix variables
    if set | grep -E -e '^_farr_KV_.*=' -e '^_farr_Kb_.*=' -e '^_farr_Ks_.*=' -e '^_farr_Vs_.*='; then
	return 1
    else
	return 0
    fi
}


#:
#: Check that no local variables are globally visible.
#:
#: Because all local variables have the ``__farr_`` prefix it can easily
#: checked that no forgotten "local" declarations exist.
#:
check_no_local_artifacts() {
    if set | grep -E -e '^__farr.*='; then
	return 1
    else
	return 0
    fi
}


#:
#: Create an new array with random entries.
#:
#: Args:
#:   $1 (str): The variable name of the array where to store the 
#:   $2 (int): The number of array entries
#:
#: Input (Globals):
#: 
create_random_array() {
    # $1 $2

    local _arr _n

    farray_create _arr
    for _n in $(/usr/bin/jot -r "${2}"  100000000 999999999); do
	farray_append _arr "${_n}"
    done
    setvar "${1}" "${_arr}"
}


#:
#: Check that an array is sorted
#:
#: Args:
#:   $1 (str): The array
#:
#: Returns:
#:   int: 0 (truthy) if the array `$1` is sorted, 1 otherwise
#:
check_array_is_sorted() {
    # $1

    local i len prev_item cur_item

    farray_length len "$1"

    [ "${len}" -le 1 ] && return 0
    farray_get prev_item "$1" 1
    i=2
    while [ "${i}" -le "${len}" ]; do
	farray_get cur_item "$1" "${i}"
	[ "${prev_item}" '>' "${cur_item}" ] && return 1
	prev_item="${cur_item}"
	i=$((i + 1))
    done
    return 0
}
