Basic tests of farray.sh's farray_XXX functions

Shell is /bin/sh.


Setup
=====

  $ set -u
  $ . "${TESTDIR}/testsetup.sh"
  $ _p_datadir="${TESTDIR}/../share/local-bsdtools"
  $ . "${_p_datadir}/farray.sh"


Basic Creation and Destruction
==============================

Create an empty array

  $ farray_create EMPTY
  $ farray_print_length EMPTY
  0 (no-eol)
  $ farray_length _var EMPTY
  $ test ${_var} -eq 0
  $ farray_istrue EMPTY
  [1]
  $ farray_debug EMPTY
  DEBUG: array `EMPTY' has length 0
  $ farray_release EMPTY
  $ check_no_array_artifacts

Create and destroy an array

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_print_length TEST
  5 (no-eol)
  $ farray_length _var TEST
  $ test ${_var} -eq 5
  $ farray_istrue TEST
  $ farray_debug TEST
  DEBUG: array `TEST' has length 5
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'90 '
#  $ farray_debug "$TEST"
#  DEBUG: array with token `[a-f0-9]+' has length 5 (re)
#  DEBUG:   the items:
#  DEBUG:     1: `0'
#  DEBUG:     2: `1'
#  DEBUG:     3: `2'
#  DEBUG:     4: `3  4   5'
#  DEBUG:     5: `" 678" \'90 '
  $ farray_release TEST
  $ check_no_array_artifacts

Create, clear and destroy an array

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_istrue TEST
  $ farray_print_length TEST
  5 (no-eol)
  $ farray_length _var TEST
  $ test ${_var} -eq 5
  $ farray_debug TEST
  DEBUG: array `TEST' has length 5
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'90 '
  $ farray_clear TEST
  $ farray_istrue TEST
  [1]
  $ farray_print_length TEST
  0 (no-eol)
  $ farray_length _var TEST
  $ test ${_var} -eq 0
  $ farray_release TEST
  $ farray_istrue TEST
  ERROR: object `TEST' not created properly: token empty
  [1]
  $ check_no_array_artifacts

Duplicate destruction

  $ farray_create TEST 1 2 3
  $ farray_release TEST
  $ farray_release TEST
  ERROR: object `TEST' not created properly: token empty
  [1]
  $ check_no_array_artifacts


Equality
========

Equality (empty)

  $ farray_create TEST1
  $ farray_create TEST2
  $ farray_are_equal TEST1 TEST2
  $ farray_release TEST1
  $ farray_release TEST2
  $ check_no_array_artifacts

Equality (non-empty)

  $ farray_create TEST1 0 1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_create TEST2 0 1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_are_equal TEST1 TEST2
  $ farray_release TEST1
  $ farray_release TEST2
  $ check_no_array_artifacts

In-Equality (non-empty, same length)

  $ farray_create TEST1 0 1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_create TEST2 0 1 2 '3  4   5' $'" 678" \\\'90'
  $ farray_are_equal TEST1 TEST2
  [1]
  $ farray_release TEST1
  $ farray_release TEST2
  $ check_no_array_artifacts

In-Equality (non-empty, different length)

  $ farray_create TEST1 0 1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_create TEST2   1 2 '3  4   5' $'" 678" \\\'90 '
  $ farray_are_equal TEST1 TEST2
  [1]
  $ farray_release TEST1
  $ farray_release TEST2
  $ check_no_array_artifacts


Get
===

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11

  $ farray_get _var TEST 1
  $ echo $_var
  0
  $ farray_tryget _var TEST 1
  $ echo $_var
  0
  $ farray_get _var TEST -5
  $ echo $_var
  0
  $ farray_tryget _var TEST -5
  $ echo $_var
  0
  $ farray_get _var TEST 6
  $ echo $_var
  11
  $ farray_tryget _var TEST 6
  $ echo $_var
  11
  $ farray_get _var TEST 0
  $ echo $_var
  11
  $ farray_tryget _var TEST 0
  $ echo $_var
  11
  $ farray_get _var TEST 5
  $ echo $_var
  " 678" \'910
  $ farray_tryget _var TEST 5
  $ echo $_var
  " 678" \'910
  $ farray_get _var TEST -1
  $ echo $_var
  " 678" \'910
  $ farray_tryget _var TEST -1
  $ echo $_var
  " 678" \'910

  $ farray_tryget _var TEST -6
  [1]
  $ ( farray_get _var TEST -6 )
  ERROR: array index out of bounds
  [70]
  $ farray_tryget _var TEST 7
  [1]
  $ ( farray_get _var TEST 7 )
  ERROR: array index out of bounds
  [70]

  $ farray_release TEST
  $ check_no_array_artifacts


Deleting
========

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_del TEST 1
  $ farray_debug TEST
  DEBUG: array `TEST' has length 5
  DEBUG:   the items:
  DEBUG:     1: `1'
  DEBUG:     2: `2'
  DEBUG:     3: `3  4   5'
  DEBUG:     4: `" 678" \'910 '
  DEBUG:     5: `11'
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_del TEST 0
  $ farray_debug TEST
  DEBUG: array `TEST' has length 5
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'910 '
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_del TEST 4
  $ farray_debug TEST
  DEBUG: array `TEST' has length 5
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `" 678" \'910 '
  DEBUG:     5: `11'
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ ( farray_del TEST "" )
  ERROR: array index out of bounds
  [70]

  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ ( farray_del TEST 7 )
  ERROR: array index out of bounds
  [70]

  $ farray_release TEST
  $ check_no_array_artifacts


Setting
=======

Append (relative)

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_set TEST "" $'the new value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 7
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'910 '
  DEBUG:     6: `11'
  DEBUG:     7: `the new value \ ''
  $ farray_release TEST
  $ check_no_array_artifacts

Append (explicit)

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_set TEST 7 $'the new value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 7
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'910 '
  DEBUG:     6: `11'
  DEBUG:     7: `the new value \ ''
  $ farray_release TEST
  $ check_no_array_artifacts

Replace the last element

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_set TEST 0 $'the new replaced value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 6
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'910 '
  DEBUG:     6: `the new replaced value \ ''
  $ farray_release TEST
  $ check_no_array_artifacts

Replace the first element

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_set TEST 1 $'the new replaced value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 6
  DEBUG:   the items:
  DEBUG:     1: `the new replaced value \ ''
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'910 '
  DEBUG:     6: `11'
  $ farray_release TEST
  $ check_no_array_artifacts

Replace some element in the middle

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_set TEST 5 $'the new replaced value \\ \' 2 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 6
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `the new replaced value \ ' 2 '
  DEBUG:     6: `11'
  $ farray_release TEST
  $ check_no_array_artifacts

Out of bounds

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ ( farray_set TEST 8 )
  ERROR: array index out of bounds (cannot create holes)
  [70]
  $ farray_release TEST
  $ check_no_array_artifacts


Inserting
=========

Append

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_insert TEST "" $'the new appended value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 7
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `1'
  DEBUG:     3: `2'
  DEBUG:     4: `3  4   5'
  DEBUG:     5: `" 678" \'910 '
  DEBUG:     6: `11'
  DEBUG:     7: `the new appended value \ ''
  $ farray_release TEST
  $ check_no_array_artifacts

Prepend

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_insert TEST 1 $'the new prepended value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 7
  DEBUG:   the items:
  DEBUG:     1: `the new prepended value \ ''
  DEBUG:     2: `0'
  DEBUG:     3: `1'
  DEBUG:     4: `2'
  DEBUG:     5: `3  4   5'
  DEBUG:     6: `" 678" \'910 '
  DEBUG:     7: `11'
  $ farray_release TEST
  $ check_no_array_artifacts

In the middle

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ farray_insert TEST 2 $'the new inserted value \\ \''
  $ farray_debug TEST
  DEBUG: array `TEST' has length 7
  DEBUG:   the items:
  DEBUG:     1: `0'
  DEBUG:     2: `the new inserted value \ ''
  DEBUG:     3: `1'
  DEBUG:     4: `2'
  DEBUG:     5: `3  4   5'
  DEBUG:     6: `" 678" \'910 '
  DEBUG:     7: `11'
  $ farray_release TEST
  $ check_no_array_artifacts

Out of bounds

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ (farray_insert TEST 8 $'the new inserted value \\ \'')
  ERROR: array index out of bounds
  [70]
  $ farray_release TEST
  $ check_no_array_artifacts

Missing value

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11
  $ (farray_insert TEST 0)
  ERROR: missing value to insert
  [70]
  $ farray_release TEST
  $ check_no_array_artifacts


Pop
===

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11

  $ farray_pop _var TEST 1
  $ echo "$_var"
  0

  $ farray_pop _var TEST 0
  $ echo "$_var"
  11
  $ farray_pop _var TEST 0
  $ printf '<%s>' "$_var"
  <" 678" \'910 > (no-eol)

  $ farray_pop _var TEST ""
  [1]

  $ farray_pop _var TEST 100
  [1]

  $ farray_release TEST
  $ check_no_array_artifacts

Special case with one element

  $ farray_create TEST 'val'
  $ farray_pop _var TEST 0
  $ echo "$_var"
  val

Special case empty array: popping from an empty list is not possible

  $ farray_pop _var TEST 0
  [1]

  $ farray_release TEST
  $ check_no_array_artifacts


Splicing
========

Replace complete array

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice "$DELETED" TEST 1 ""  "$INSERTED"
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 9
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  DEBUG:     9: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 3
  DEBUG:   the items:
  DEBUG:     1: ` I 1 '
  DEBUG:     2: ` I 2 '
  DEBUG:     3: ` I 3 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Insert at the beginning (prepend)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 0  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 0
  $ farray_debug TEST
  DEBUG: array `TEST' has length 12
  DEBUG:   the items:
  DEBUG:     1: ` I 1 '
  DEBUG:     2: ` I 2 '
  DEBUG:     3: ` I 3 '
  DEBUG:     4: ` v 1 '
  DEBUG:     5: ` v 2 '
  DEBUG:     6: ` v 3 '
  DEBUG:     7: ` v 4 '
  DEBUG:     8: ` v 5 '
  DEBUG:     9: ` v 6 '
  DEBUG:     10: ` v 7 '
  DEBUG:     11: ` v 8 '
  DEBUG:     12: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Insert at the end (extend)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST "" 0  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 0
  $ farray_debug TEST
  DEBUG: array `TEST' has length 12
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  DEBUG:     9: ` v 9 '
  DEBUG:     10: ` I 1 '
  DEBUG:     11: ` I 2 '
  DEBUG:     12: ` I 3 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Insert in the middle

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 4 0  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 0
  $ farray_debug TEST
  DEBUG: array `TEST' has length 12
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` I 1 '
  DEBUG:     5: ` I 2 '
  DEBUG:     6: ` I 3 '
  DEBUG:     7: ` v 4 '
  DEBUG:     8: ` v 5 '
  DEBUG:     9: ` v 6 '
  DEBUG:     10: ` v 7 '
  DEBUG:     11: ` v 8 '
  DEBUG:     12: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

"Clear" (empty inserted)

  $ farray_create DELETED
  $ farray_create INSERTED
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 "" INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 9
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  DEBUG:     9: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 0
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

"Clear" (no inserted)

  $ farray_create DELETED
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 ""
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 9
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  DEBUG:     9: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 0
  $ farray_release DELETED
  $ farray_release TEST
  $ check_no_array_artifacts

Cut out some part in the middle

  $ farray_create DELETED
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 2 5
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 5
  DEBUG:   the items:
  DEBUG:     1: ` v 2 '
  DEBUG:     2: ` v 3 '
  DEBUG:     3: ` v 4 '
  DEBUG:     4: ` v 5 '
  DEBUG:     5: ` v 6 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 4
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 7 '
  DEBUG:     3: ` v 8 '
  DEBUG:     4: ` v 9 '
  $ farray_release DELETED
  $ farray_release TEST
  $ check_no_array_artifacts

Cut out some part from the beginning

  $ farray_create DELETED
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 6
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 6
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 3
  DEBUG:   the items:
  DEBUG:     1: ` v 7 '
  DEBUG:     2: ` v 8 '
  DEBUG:     3: ` v 9 '
  $ farray_release DELETED
  $ farray_release TEST
  $ check_no_array_artifacts

Cut out some part at the end (1)

  $ farray_create DELETED
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 0 1
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 1
  DEBUG:   the items:
  DEBUG:     1: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 8
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  $ farray_release DELETED
  $ farray_release TEST
  $ check_no_array_artifacts

Cut out some part at the end (1)

  $ farray_create DELETED
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST -1 ""
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 2
  DEBUG:   the items:
  DEBUG:     1: ` v 8 '
  DEBUG:     2: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 7
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  $ farray_release DELETED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice at the beginning (shrink)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 4  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 4
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 8
  DEBUG:   the items:
  DEBUG:     1: ` I 1 '
  DEBUG:     2: ` I 2 '
  DEBUG:     3: ` I 3 '
  DEBUG:     4: ` v 5 '
  DEBUG:     5: ` v 6 '
  DEBUG:     6: ` v 7 '
  DEBUG:     7: ` v 8 '
  DEBUG:     8: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice at the end (shrink)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST -3 4  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 4
  DEBUG:   the items:
  DEBUG:     1: ` v 6 '
  DEBUG:     2: ` v 7 '
  DEBUG:     3: ` v 8 '
  DEBUG:     4: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 8
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` I 1 '
  DEBUG:     7: ` I 2 '
  DEBUG:     8: ` I 3 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice in the middle (shrink)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 3 4  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 4
  DEBUG:   the items:
  DEBUG:     1: ` v 3 '
  DEBUG:     2: ` v 4 '
  DEBUG:     3: ` v 5 '
  DEBUG:     4: ` v 6 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 8
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` I 1 '
  DEBUG:     4: ` I 2 '
  DEBUG:     5: ` I 3 '
  DEBUG:     6: ` v 7 '
  DEBUG:     7: ` v 8 '
  DEBUG:     8: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice at the beginning (grow)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 2  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 2
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 10
  DEBUG:   the items:
  DEBUG:     1: ` I 1 '
  DEBUG:     2: ` I 2 '
  DEBUG:     3: ` I 3 '
  DEBUG:     4: ` v 3 '
  DEBUG:     5: ` v 4 '
  DEBUG:     6: ` v 5 '
  DEBUG:     7: ` v 6 '
  DEBUG:     8: ` v 7 '
  DEBUG:     9: ` v 8 '
  DEBUG:     10: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice at the end (grow)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST -1 2  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 2
  DEBUG:   the items:
  DEBUG:     1: ` v 8 '
  DEBUG:     2: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 10
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` I 1 '
  DEBUG:     9: ` I 2 '
  DEBUG:     10: ` I 3 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice in the middle (grow)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 3 2  INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 2
  DEBUG:   the items:
  DEBUG:     1: ` v 3 '
  DEBUG:     2: ` v 4 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 10
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` I 1 '
  DEBUG:     4: ` I 2 '
  DEBUG:     5: ` I 3 '
  DEBUG:     6: ` v 5 '
  DEBUG:     7: ` v 6 '
  DEBUG:     8: ` v 7 '
  DEBUG:     9: ` v 8 '
  DEBUG:     10: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice at the beginning (no length change)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 1 3 INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 3
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 9
  DEBUG:   the items:
  DEBUG:     1: ` I 1 '
  DEBUG:     2: ` I 2 '
  DEBUG:     3: ` I 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  DEBUG:     9: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice at the end (no length change)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST -2 3 INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 3
  DEBUG:   the items:
  DEBUG:     1: ` v 7 '
  DEBUG:     2: ` v 8 '
  DEBUG:     3: ` v 9 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 9
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` v 4 '
  DEBUG:     5: ` v 5 '
  DEBUG:     6: ` v 6 '
  DEBUG:     7: ` I 1 '
  DEBUG:     8: ` I 2 '
  DEBUG:     9: ` I 3 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts

Splice in the middle (no length change)

  $ farray_create DELETED
  $ farray_create INSERTED  ' I 1 ' ' I 2 ' ' I 3 '
  $ farray_create TEST ' v 1 ' ' v 2 ' ' v 3 ' ' v 4 ' ' v 5 ' ' v 6 ' ' v 7 ' ' v 8 '  ' v 9 '
  $ farray_splice DELETED TEST 4 3 INSERTED
  $ farray_debug DELETED
  DEBUG: array `DELETED' has length 3
  DEBUG:   the items:
  DEBUG:     1: ` v 4 '
  DEBUG:     2: ` v 5 '
  DEBUG:     3: ` v 6 '
  $ farray_debug TEST
  DEBUG: array `TEST' has length 9
  DEBUG:   the items:
  DEBUG:     1: ` v 1 '
  DEBUG:     2: ` v 2 '
  DEBUG:     3: ` v 3 '
  DEBUG:     4: ` I 1 '
  DEBUG:     5: ` I 2 '
  DEBUG:     6: ` I 3 '
  DEBUG:     7: ` v 7 '
  DEBUG:     8: ` v 8 '
  DEBUG:     9: ` v 9 '
  $ farray_release DELETED
  $ farray_release INSERTED
  $ farray_release TEST
  $ check_no_array_artifacts


Searching
=========

Special case empty array because of start/stop indexes

  $ farray_create TEST
  $ farray_find _var TEST foo
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts


  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11

  $ farray_find _var TEST 0
  $ echo $_var
  1
  $ farray_find _var TEST $'" 678" \\\'910 '
  $ echo $_var
  5
  $ farray_find _var TEST '11'
  $ echo $_var
  6
  $ farray_find _var TEST $'" 678" \\\'910 ' "" ""
  $ echo $_var
  5
  $ farray_find _var TEST '11' 1 5
  [1]

  $ farray_find _var TEST '11' 7
  [1]

  $ farray_find _var TEST '11' 7 6
  [1]

  $ farray_find _var TEST '11' 1 ""
  $ echo $_var
  6

  $ ( farray_find _var TEST '11' 1 7 )
  ERROR: end index exceeds array length
  [70]

  $ ( farray_find _var TEST '11' -6 )
  ERROR: start index must be >= 1
  [70]

  $ ( farray_find _var TEST '11' 1 -6 )
  ERROR: end index must be >= 1
  [70]

  $ farray_contains TEST '0'
  $ farray_contains TEST $'" 678" \\\'910 '
  $ farray_contains TEST '11'

  $ farray_contains TEST '0 '
  [1]

  $ farray_release TEST
  $ check_no_array_artifacts


Iterating
=========

  $ farray_create TEST 0 1 2 '3  4   5' $'" 667788" \\\'910 ' 11

MANUAL (by name)

  $ _v=''
  > _i=1
  > while farray_tryget _v TEST ${_i}; do
  >     printf "  IDX: \`%d', VAL: \`%s'\\n" "${_i}" "${_v}"
  >     _i=$((_i + 1))
  > done
    IDX: `1', VAL: `0'
    IDX: `2', VAL: `1'
    IDX: `3', VAL: `2'
    IDX: `4', VAL: `3  4   5'
    IDX: `5', VAL: `" 667788" \'910 '
    IDX: `6', VAL: `11'

MANUAL (by value, reversed, using negative indexes)

  $ _v=''
  > _i=0
  > while farray_tryget _v "$TEST" ${_i}; do
  >     printf "  IDX: \`%d', VAL: \`%s'\\n" "${_i}" "${_v}"
  >     _i=$((_i - 1))
  > done
    IDX: `0', VAL: `11'
    IDX: `-1', VAL: `" 667788" \'910 '
    IDX: `-2', VAL: `3  4   5'
    IDX: `-3', VAL: `2'
    IDX: `-4', VAL: `1'
    IDX: `-5', VAL: `0'

ITERATE (for each, by name)

  $ farray_for_each TEST $'printf "EACH: %s at idx %d: value is \\`%s\\\'\\n"'
  EACH: TEST at idx 1: value is `0'
  EACH: TEST at idx 2: value is `1'
  EACH: TEST at idx 3: value is `2'
  EACH: TEST at idx 4: value is `3  4   5'
  EACH: TEST at idx 5: value is `" 667788" \'910 '
  EACH: TEST at idx 6: value is `11'

ITERATE (for each, by value)

  $ farray_for_each "$TEST" $'printf "EACH: %s at idx %d: value is \\`%s\\\'\\n"'
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 1: value is `0' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 2: value is `1' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 3: value is `2' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 4: value is `3  4   5' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 5: value is `" 667788" \\'910 ' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 6: value is `11' (re)

REVERSE ITERATE (for each, by name)

  $ farray_reversed_for_each TEST $'printf "EACH: %s at idx %d: value is \\`%s\\\'\\n"'
  EACH: TEST at idx 6: value is `11'
  EACH: TEST at idx 5: value is `" 667788" \'910 '
  EACH: TEST at idx 4: value is `3  4   5'
  EACH: TEST at idx 3: value is `2'
  EACH: TEST at idx 2: value is `1'
  EACH: TEST at idx 1: value is `0'

ITERATE (for each, by value)

  $ farray_reversed_for_each "$TEST" $'printf "EACH: %s at idx %d: value is \\`%s\\\'\\n"'
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 6: value is `11' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 5: value is `" 667788" \\'910 ' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 4: value is `3  4   5' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 3: value is `2' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 2: value is `1' (re)
  EACH: _farr_A\[\?\]:[a-f0-9]+ at idx 1: value is `0' (re)

  $ farray_release TEST
  $ check_no_array_artifacts


Eval / Quoting
==============

# Use Dollar-Single-Quotes
  $ _var1=$'" 678" \\\'910 '
# Use just Single-Quotes
  $ _var2='" 678" \'\''910 '
# Use a Double-Quotes to insert a single quote
  $ _var3='" 678" \'"'"'910 '
  $ test "${_var1}" = "${_var2}"
  $ test "${_var1}" = "${_var3}"
  $ eval _evar="\$(_farr_quote_for_eval \"\${_var1}\")"
  $ eval printf '%s' "${_evar}"
  " 678" \'910  (no-eol)
  $ _var1=$'" 678" \\\'910\t\''
  $ _var2='" 678" \'\''910	'\'''
  $ _var3='" 678" \'"'"'910	'"'"''
  $ test "${_var1}" = "${_var2}"
  $ test "${_var1}" = "${_var3}"
  $ eval _evar="\$(_farr_quote_for_eval \"\${_var1}\")"
  $ eval printf '%s' "${_evar}"
  " 678" \'910	' (no-eol)
  $ _var1=$'" 678" \\\'910\t\''
  $ eval _evar="\$(_farr_quote_for_eval_strict \"\${_var1}\")"
  $ eval printf '%s' "${_evar}"
  " 678" \'910	' (no-eol)


Join
====

  $ farray_create TEST 0 1 2 '3  4   5' $'" 678" \\\'910 ' 11

  $ farray_join _var TEST
  $ echo "$_var"
  0 1 2 3  4   5 " 678" \'910  11

  $ farray_join _var TEST '---'
  $ echo "$_var"
  0---1---2---3  4   5---" 678" \'910 ---11

  $ farray_join_for_eval _var TEST
  $ echo "$_var"
  '0' '1' '2' '3  4   5' $'" 678" \\\'910 ' '11'

  $ farray_print_join_for_eval TEST
  '0' '1' '2' '3  4   5' $'" 678" \\\'910 ' '11' (no-eol)

  $ farray_clear TEST

  $ farray_join _var TEST
  $ echo "/${_var}/"
  //

  $ farray_print_join_for_eval TEST

  $ farray_join_for_eval _var TEST
  $ echo "!${_var}!"
  !!

  $ farray_join _var TEST '---'
  $ echo "/${_var}/"
  //
  $ farray_release TEST
  $ check_no_array_artifacts


Parameter Checks
================

  $ ( farray_create )
  ERROR: missing farray name
  [70]

  $ ( farray_append )
  ERROR: missing farray name or token value
  [70]

  $ TEST=''
  $ (farray_append TEST)
  ERROR: object `TEST' not created properly: token empty
  [70]

  $ check_no_array_artifacts

  $ farray_create DUP
  $ (farray_create DUP)
  ERROR: object `DUP' already created \(value `_farr_A\[\?\]:[a-f0-9]+'\) (re)
  [70]

  $ check_no_array_artifacts
  _farr_A_[0-9a-f]+_C\=1 (re)
  _farr_A_[0-9a-f]+__\=0 (re)
  [1]

  $ farray_release DUP
  $ check_no_array_artifacts


Cross Type Checks
=================

  $ farray_create TEST
  $ (falist_set TEST key value)
  ERROR: object `TEST' is not an alist
  [70]
  $ farray_release TEST
  $ check_no_array_artifacts


Other Type Checks
=================

  $ farray_type
  unknown (no-eol)
  $ farray_type __UNKNOWN_VARIABLE__
  unknown (no-eol)
  $ testvar='foo'
  $ farray_type testvar
  value (no-eol)
  $ testvar=''
  $ farray_type testvar
  null (no-eol)
  $ farray_isarray testvar
  [1]
  $ farray_create TEST
  $ falist_type TEST
  array (no-eol)
  $ farray_isarray TEST

  $ farray_isref "$TEST"
  $ farray_isref TEST
  [1]
  $ farray_isref ''
  [1]

  $ farray_release TEST
  $ check_no_array_artifacts


Merge
=====

  $ farray_create MERGED
  $ farray_create INPUT1 s1 s2 s3
  $ farray_create INPUT2
  $ farray_merge MERGED INPUT1 INPUT2
  $ farray_release INPUT1
  $ farray_release INPUT2
  $ farray_debug MERGED
  DEBUG: array `MERGED' has length 3
  DEBUG:   the items:
  DEBUG:     1: `s1'
  DEBUG:     2: `s2'
  DEBUG:     3: `s3'
  $ farray_release MERGED
  $ check_no_array_artifacts

  $ farray_create MERGED
  $ farray_create INPUT1
  $ farray_create INPUT2 S1 S2 S3 S4
  $ farray_merge MERGED INPUT1 INPUT2
  $ farray_release INPUT1
  $ farray_release INPUT2
  $ farray_debug MERGED
  DEBUG: array `MERGED' has length 4
  DEBUG:   the items:
  DEBUG:     1: `S1'
  DEBUG:     2: `S2'
  DEBUG:     3: `S3'
  DEBUG:     4: `S4'
  $ farray_release MERGED
  $ check_no_array_artifacts

  $ farray_create MERGED
  $ farray_create INPUT1 s1 s2 s3 s4
  $ farray_create INPUT2 S1 S2 S3 S4
  $ farray_merge MERGED INPUT1 INPUT2
  $ farray_release INPUT1
  $ farray_release INPUT2
  $ farray_debug MERGED
  DEBUG: array `MERGED' has length 8
  DEBUG:   the items:
  DEBUG:     1: `S1'
  DEBUG:     2: `S2'
  DEBUG:     3: `S3'
  DEBUG:     4: `S4'
  DEBUG:     5: `s1'
  DEBUG:     6: `s2'
  DEBUG:     7: `s3'
  DEBUG:     8: `s4'
  $ farray_release MERGED
  $ check_no_array_artifacts

  $ farray_create MERGED
  $ farray_create INPUT1 s1 s2 s3 s4
  $ farray_create INPUT2 S1 S2 S3 S4
  $ farray_merge MERGED INPUT2 INPUT1
  $ farray_release INPUT1
  $ farray_release INPUT2
  $ farray_debug MERGED
  DEBUG: array `MERGED' has length 8
  DEBUG:   the items:
  DEBUG:     1: `S1'
  DEBUG:     2: `S2'
  DEBUG:     3: `S3'
  DEBUG:     4: `S4'
  DEBUG:     5: `s1'
  DEBUG:     6: `s2'
  DEBUG:     7: `s3'
  DEBUG:     8: `s4'
  $ farray_release MERGED
  $ check_no_array_artifacts

  $ farray_create MERGED
  $ farray_create INPUT1 S1 S2 S3 S4
  $ farray_create INPUT2 s1 s2 s3 s4
  $ farray_merge "$MERGED" "$INPUT1" "$INPUT2"
  $ farray_release INPUT1
  $ farray_release INPUT2
  $ farray_debug MERGED
  DEBUG: array `MERGED' has length 8
  DEBUG:   the items:
  DEBUG:     1: `S1'
  DEBUG:     2: `S2'
  DEBUG:     3: `S3'
  DEBUG:     4: `S4'
  DEBUG:     5: `s1'
  DEBUG:     6: `s2'
  DEBUG:     7: `s3'
  DEBUG:     8: `s4'
  $ farray_release MERGED
  $ check_no_array_artifacts

  $ farray_create MERGED
  $ farray_create INPUT1 S1 S2 S3 S4
  $ farray_create INPUT2 s1 s2 s3 s4
  $ farray_merge "$MERGED" "$INPUT2" "$INPUT1"
  $ farray_release INPUT1
  $ farray_release INPUT2
  $ farray_debug MERGED
  DEBUG: array `MERGED' has length 8
  DEBUG:   the items:
  DEBUG:     1: `S1'
  DEBUG:     2: `S2'
  DEBUG:     3: `S3'
  DEBUG:     4: `S4'
  DEBUG:     5: `s1'
  DEBUG:     6: `s2'
  DEBUG:     7: `s3'
  DEBUG:     8: `s4'
  $ farray_release MERGED
  $ check_no_array_artifacts


Sort
====

  $ farray_create TEST 5 3 2 4
  $ farray_sort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_debug TEST
  DEBUG: array `TEST' has length 4
  DEBUG:   the items:
  DEBUG:     1: `2'
  DEBUG:     2: `3'
  DEBUG:     3: `4'
  DEBUG:     4: `5'
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 5 3 2 4
  $ farray_insertionsort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_debug TEST
  DEBUG: array `TEST' has length 4
  DEBUG:   the items:
  DEBUG:     1: `2'
  DEBUG:     2: `3'
  DEBUG:     3: `4'
  DEBUG:     4: `5'
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 5 3 2 4
  $ farray_heapsort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_debug TEST
  DEBUG: array `TEST' has length 4
  DEBUG:   the items:
  DEBUG:     1: `2'
  DEBUG:     2: `3'
  DEBUG:     3: `4'
  DEBUG:     4: `5'
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 5 3 2 4
  $ farray_heapsort_bottomup TEST
  $ check_array_is_sorted "$TEST"
  $ farray_debug TEST
  DEBUG: array `TEST' has length 4
  DEBUG:   the items:
  DEBUG:     1: `2'
  DEBUG:     2: `3'
  DEBUG:     3: `4'
  DEBUG:     4: `5'
  $ farray_release TEST
  $ check_no_array_artifacts

  $ create_random_array UNSORTED 1000
  $ check_array_is_sorted "$UNSORTED"
  [1]

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ check_array_is_sorted "$TEST"
  [1]
  $ farray_gnomesort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ check_array_is_sorted "$TEST"
  [1]
  $ farray_insertionsort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ check_array_is_sorted "$TEST"
  [1]
  $ farray_shellsort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ check_array_is_sorted "$TEST"
  [1]
  $ farray_heapsort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ check_array_is_sorted "$TEST"
  [1]
  $ farray_heapsort_bottomup TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_release UNSORTED
  $ check_no_array_artifacts

# Extra checks for Heapsort

  $ farray_create UNSORTED '189548216' '544226607' '690563482' '224884577' '843524724' '922143089' '917031008' '602352555' '397442038' '350475285'

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ farray_heapsort TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_create TEST
  $ farray_splice "" TEST 1 "" UNSORTED
  $ farray_heapsort_bottomup TEST
  $ check_array_is_sorted "$TEST"
  $ farray_release TEST

  $ farray_release UNSORTED
  $ check_no_array_artifacts


Binary Search
=============

  $ farray_create TEST
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_binsearch _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch _var TEST 5
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_binsearch _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch _var TEST 5
  [1]
  $ farray_binsearch _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch _var TEST 7
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_binsearch _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch _var TEST 5
  [1]
  $ farray_binsearch _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch _var TEST 7
  [1]
  $ farray_binsearch _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch _var TEST 9
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8 88
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_binsearch _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch _var TEST 5
  [1]
  $ farray_binsearch _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch _var TEST 7
  [1]
  $ farray_binsearch _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch _var TEST 87
  [1]
  $ farray_binsearch _var TEST 88
  $ echo ${_var}
  5
  $ farray_binsearch _var TEST 888
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8 88 8888
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_binsearch _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch _var TEST 5
  [1]
  $ farray_binsearch _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch _var TEST 7
  [1]
  $ farray_binsearch _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch _var TEST 87
  [1]
  $ farray_binsearch _var TEST 88
  $ echo ${_var}
  5
  $ farray_binsearch _var TEST 888
  [1]
  $ farray_binsearch _var TEST 8888
  $ echo ${_var}
  6
  $ farray_binsearch _var TEST 88888
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8 88 8888 9
  $ farray_binsearch _var TEST 1
  [1]
  $ farray_binsearch _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch _var TEST 3
  [1]
  $ farray_binsearch _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch _var TEST 5
  [1]
  $ farray_binsearch _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch _var TEST 7
  [1]
  $ farray_binsearch _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch _var TEST 87
  [1]
  $ farray_binsearch _var TEST 88
  $ echo ${_var}
  5
  $ farray_binsearch _var TEST 888
  [1]
  $ farray_binsearch _var TEST 8888
  $ echo ${_var}
  6
  $ farray_binsearch _var TEST 88888
  [1]
  $ farray_binsearch _var TEST 9
  $ echo ${_var}
  7
  $ farray_binsearch _var TEST 99
  [1]
  $ farray_release TEST
  $ check_no_array_artifacts


Binary Search Leftmost
======================

  $ farray_create TEST
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 5
  $ echo ${_var}
  3
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 5
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 7
  $ echo ${_var}
  4
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 5
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 7
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 9
  $ echo ${_var}
  5
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8 88
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 5
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 7
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 87
  $ echo ${_var}
  5
  $ farray_binsearch_leftmost _var TEST 88
  $ echo ${_var}
  5
  $ farray_binsearch_leftmost _var TEST 888
  $ echo ${_var}
  6
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8 88 8888
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 5
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 7
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 87
  $ echo ${_var}
  5
  $ farray_binsearch_leftmost _var TEST 88
  $ echo ${_var}
  5
  $ farray_binsearch_leftmost _var TEST 888
  $ echo ${_var}
  6
  $ farray_binsearch_leftmost _var TEST 8888
  $ echo ${_var}
  6
  $ farray_binsearch_leftmost _var TEST 88888
  $ echo ${_var}
  7
  $ farray_release TEST
  $ check_no_array_artifacts

  $ farray_create TEST 2 4 6 8 88 8888 9
  $ farray_binsearch_leftmost _var TEST 1
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 2
  $ echo ${_var}
  1
  $ farray_binsearch_leftmost _var TEST 3
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 4
  $ echo ${_var}
  2
  $ farray_binsearch_leftmost _var TEST 5
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 6
  $ echo ${_var}
  3
  $ farray_binsearch_leftmost _var TEST 7
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 8
  $ echo ${_var}
  4
  $ farray_binsearch_leftmost _var TEST 87
  $ echo ${_var}
  5
  $ farray_binsearch_leftmost _var TEST 88
  $ echo ${_var}
  5
  $ farray_binsearch_leftmost _var TEST 888
  $ echo ${_var}
  6
  $ farray_binsearch_leftmost _var TEST 8888
  $ echo ${_var}
  6
  $ farray_binsearch_leftmost _var TEST 88888
  $ echo ${_var}
  7
  $ farray_binsearch_leftmost _var TEST 9
  $ echo ${_var}
  7
  $ farray_binsearch_leftmost _var TEST 99
  $ echo ${_var}
  8
  $ farray_release TEST
  $ check_no_array_artifacts


Generic Destruction
===================

  $ farray_create TEST i1 i2
  $ farray_debug TEST
  DEBUG: array `TEST' has length 2
  DEBUG:   the items:
  DEBUG:     1: `i1'
  DEBUG:     2: `i2'
  $ _farr_release_object "$TEST"
  $ check_no_array_artifacts
  $ (_farr_release_object "$TEST")
  ERROR: farray `_farr_A\[\?\]:([a-f0-9]+)' not created properly: no storage for token `\1' (re)
  [1]
  $ TEST=''
  $ _farr_release_object "$TEST"
  $ check_no_array_artifacts


Complex Debug
=============

  $ farray_create TEST i1 i2
  $ farray_create ITEM1 i11 i22
  $ farray_append TEST "$ITEM1"
  $ farray_release ITEM1
  $ farray_debug TEST
  DEBUG: array `TEST' has length 3
  DEBUG:   the items:
  DEBUG:     1: `i1'
  DEBUG:     2: `i2'
  DEBUG:     3: >>>
      DEBUG: array with token `[a-f0-9]+' has length 2 (re)
      DEBUG:   the items:
      DEBUG:     1: `i11'
      DEBUG:     2: `i22'
  $ farray_release TEST
  $ check_no_array_artifacts


No Locals
=========

  $ check_no_local_artifacts
